# -*- coding: utf-8 -*-
from passport.backend.core.db.runner import run_incr_id_query
from passport.backend.core.differ.types import (
    Diff,
    EmptyDiff,
)
from passport.backend.core.differ.utils import slice_diff
from passport.backend.core.serializers.eav.base import EavSerializer
from passport.backend.core.serializers.eav.extended_attributes import EAV_EXTENDED_ATTRIBUTES_WEBAUTHN_CREDENTIAL_MAPPER
from passport.backend.core.serializers.eav.query import (
    EavDeleteAllWebauthnCredentialsQuery,
    EavDeleteAllWebauthnCredentialToUidQuery,
    EavDeleteWebauthnCredentialQuery,
    EavDeleteWebauthnCredentialToUidQuery,
    EavInsertWebauthnCredentialToUidQuery,
)
from passport.backend.core.undefined import Undefined


class WebauthnCredentialEavSerializer(EavSerializer):
    def serialize(self, old, new, difference):
        if not old and not new:
            return

        if new and not new.external_id:
            raise ValueError('WebauthnCredential without external_id cannot be serialized')
        if old and old.external_id and new and new.external_id and old.external_id != new.external_id:
            raise ValueError('WebauthnCredential\'s external_id is immutable')

        creds = old.parent if old else new.parent
        uid = creds.parent.uid

        if old and not new:
            yield EavDeleteWebauthnCredentialToUidQuery(uid, old.external_id)
            yield EavDeleteWebauthnCredentialQuery(uid, old.id)
            return

        # Если у нас ещё нет ID, то сгенерируем его
        if new.id is Undefined:
            query = EavSerializer().build_increment_webauthn_credential_id_query()
            new.id = run_incr_id_query(query)

        # Выполняем стандартную сериализацию сущностей
        for query in self.build_extended_attribute_queries(
            EAV_EXTENDED_ATTRIBUTES_WEBAUTHN_CREDENTIAL_MAPPER,
            uid, old, new,
            difference.get_changed_fields(EAV_EXTENDED_ATTRIBUTES_WEBAUTHN_CREDENTIAL_MAPPER),
            create=old is None,
        ):
            yield query

        if not old:
            yield EavInsertWebauthnCredentialToUidQuery(uid, new.external_id)


class WebauthnCredentialsEavSerializer(EavSerializer):
    def serialize(self, old, new, difference):
        if difference == EmptyDiff:
            return

        if difference.deleted is None:
            difference = Diff(difference.added, difference.changed, {})

        if not new and old and old.all():
            yield EavDeleteAllWebauthnCredentialToUidQuery(old.parent.uid)
            yield EavDeleteAllWebauthnCredentialsQuery(old.parent.uid)
            return

        creds_diff = slice_diff(difference, '_creds_by_cred_id')
        for cred_external_id in creds_diff.get_changed_fields():
            old_cred = old.by_external_id(cred_external_id) if old else None
            new_cred = new.by_external_id(cred_external_id) if new else None

            instance_diff = slice_diff(creds_diff, cred_external_id)

            for query in WebauthnCredentialEavSerializer().serialize(
                old_cred,
                new_cred,
                instance_diff,
            ):
                yield query
