# -*- coding: utf-8 -*-

from passport.backend.core.db.schemas import (
    family_info_table,
    family_members_table,
)
from passport.backend.core.models.family import FamilyInfo
from passport.backend.core.serializers.base import DirectSerializer
from passport.backend.core.serializers.query import (
    GenericDeleteQuery,
    GenericInsertQuery,
    GenericQuery,
)
from passport.backend.core.serializers.utils import CallbackResultContainer
from passport.backend.core.undefined import Undefined
from sqlalchemy.sql import (
    and_,
    literal_column,
    select,
)


class CreateFamilyQuery(GenericQuery):
    def __init__(self, admin_uid, meta):
        super(CreateFamilyQuery, self).__init__(family_info_table, {})
        self.admin_uid = admin_uid
        self.meta = meta

    def to_query(self):
        values = {
            'admin_uid': self.admin_uid,
            'meta': self.meta,
        }
        return self.table.insert(values)


class DeleteFamilyMemberQuery(GenericQuery):
    def __init__(self, family_id, uid):
        super(DeleteFamilyMemberQuery, self).__init__(family_members_table, {})
        self.table_info = family_info_table
        self.family_id = family_id
        self.uid = uid

    def to_query(self):
        return self.table.delete().where(
            and_(
                self.table.c.family_id == self.family_id,
                self.table.c.uid == self.uid,
            ),
        )


class AddFamilyMemberWithCheckQuery(GenericQuery):
    def __init__(self, family_id, uid, place, admin_uid):
        super(AddFamilyMemberWithCheckQuery, self).__init__(family_members_table, {})
        self.table_info = family_info_table
        self.family_id = family_id
        self.uid = uid
        self.place = place
        self.admin_uid = admin_uid

    def to_query(self):
        return self.table.insert().from_select(
            self.table.c.keys(),
            select([
                self.table_info.c.family_id,
                literal_column(str(self.uid)),
                literal_column(str(self.place)),
            ]).where(
                and_(
                    self.table_info.c.family_id == self.family_id,
                    self.table_info.c.admin_uid == self.admin_uid,
                ),
            ),
        )


class FamilyInfoSerializer(DirectSerializer):
    model = FamilyInfo
    table = family_info_table

    field_mapping = {
        '_family_id': 'family_id',
        'admin_uid': 'admin_uid',
    }

    def extract_data(self, old, new, fields):
        data = super(FamilyInfoSerializer, self).extract_data(old, new, fields)
        if '_family_id' in data:
            data['family_id'] = data.pop('_family_id')
        return data

    @staticmethod
    def _add_member_query(family_info, uid, place, check=False):
        if check:
            return AddFamilyMemberWithCheckQuery(
                family_id=family_info._family_id,
                uid=uid,
                place=place,
                admin_uid=family_info.admin_uid,
            )
        else:
            return GenericInsertQuery(
                family_members_table,
                {
                    'family_id': family_info._family_id,
                    'uid': uid,
                    'place': place,
                },
            )

    @staticmethod
    def _delete_member_query(family_info, uid):
        return DeleteFamilyMemberQuery(
            family_info._family_id,
            uid,
        )

    def create(self, old, new, data):
        members = data.pop('members')
        kids = data.pop('kids', Undefined)

        q_create_family = CreateFamilyQuery(
            data['admin_uid'],
            b'',
        )

        yield (
            q_create_family,
            lambda result: setattr(new, 'family_id', result.inserted_primary_key[0]),
        )

        if members != Undefined:
            for member in new.members.values():
                yield self._add_member_query(new, member.uid, member.place)

        if kids and kids.members:
            for kid in new.kids.members.values():
                yield self._add_member_query(new, kid.uid, kid.place)

    def change(self, old, new, data):
        members = data.pop('members', Undefined)
        kids = data.pop('kids', Undefined)

        # Применяем изменения семьи
        if data:
            for query in super(FamilyInfoSerializer, self).change(old, new, data):
                yield query

        # Применяем изменения списка членов семьи
        if members is not Undefined:
            for query in self._get_change_member_collection_queries(old, new, new):
                yield query
        if kids and kids.members:
            for query in self._get_change_member_collection_queries(old.kids, new.kids, new):
                yield query

    def _get_change_member_collection_queries(self, old, new, family_info):
        old_members_uids = set(v.uid for v in old.members.values())
        new_members_uids = set(v.uid for v in new.members.values())
        uids_to_delete = old_members_uids - new_members_uids
        for uid in uids_to_delete:
            yield self._delete_member_query(family_info, uid)

        uids_to_create = new_members_uids - old_members_uids
        for uid in uids_to_create:
            yield self._add_member_query(
                family_info=family_info,
                uid=uid,
                place=new.members[uid].place,
                check=True,
            )

    def delete(self, old):
        # Удаляем семью
        q_delete_family_info = GenericDeleteQuery(
            self.table,
            None,
            self.table.c.family_id == old._family_id,
        )
        result = CallbackResultContainer()
        yield (
            q_delete_family_info,
            result,
        )

        # Удаляем членов семьи, если семья удалилась
        if result.val.rowcount:
            yield GenericDeleteQuery(
                family_members_table,
                None,
                family_members_table.c.family_id == old._family_id,
            )
