# -*- coding: utf-8 -*-

from copy import deepcopy

from passport.backend.core.differ import slice_diff
from passport.backend.core.serializers.logs.utils import (
    hierarchical_exists,
    hierarchical_get,
    is_not_empty,
    join_str,
)
from passport.backend.utils.common import unique_unhashable


ADDED = 'added'
CHANGED = 'changed'
DELETED = 'deleted'


class BaseSerializerRule(object):
    """Базовое правило сериализации изменений аккаунта для записи в логи"""
    def __init__(self, entity, entity_alias=None,
                 operations=(ADDED, CHANGED, DELETED), model_attr=None):
        """
        entity: поле, за изменением которого следим.
        entity_alias: имя поля для записи в лог. По умолчанию совпадает с entity.
        operations: операции, для которых применимо правило.
        model_attr: атрибут модели, с которого берётся значение.
        """
        self.entity = entity
        self._entity_alias = entity_alias
        self._model_attr = model_attr
        self.allowed_operations = operations

    @property
    def entity_alias(self):
        return self._entity_alias or self.entity

    @property
    def model_attr(self):
        return self._model_attr or self.entity

    def make_entries(self, old_snapshot, new_snapshot, diff):
        """Возвращает список лог-записей, созданных этим правилом"""
        raise NotImplementedError()  # pragma: no cover

    def is_applicable(self, old_snapshot, new_snapshot, diff):
        """Применимо ли правило к данному диффу"""
        if any(
            hierarchical_exists(diff, '%s.%s' % (operation, self.entity))
            for operation in self.allowed_operations
        ):
            old_value = hierarchical_get(old_snapshot, self.entity)
            new_value = hierarchical_get(new_snapshot, self.entity)
            return is_not_empty(old_value) or is_not_empty(new_value)  # ничего не пишем при замене None на ''

    def clone_for_prefix(self, entity_prefix):
        """
        Создаёт новое правило, применимое для данного префикса
        entity_prefix - точкоразделённая строка, путь от аккаунта до сериализуемой модели
        """
        new_rule = deepcopy(self)
        new_rule.entity = join_str(entity_prefix, new_rule.entity)
        return new_rule


class BaseModelSerializer(object):
    """Базовый класс сериализатора модели"""
    entity = None
    rules = None
    child_serializers = None

    def __init__(self, entity=None):
        self.entity = entity

    def serialize_rules(self, old_snapshot, new_snapshot, diff, rules, new_entity_prefix,
                        check_is_applicable=True, **kwargs):
        entries = []

        for rule in rules:
            rule = rule.clone_for_prefix(new_entity_prefix)
            if not check_is_applicable or rule.is_applicable(old_snapshot, new_snapshot, diff):
                entries += rule.make_entries(old_snapshot, new_snapshot, diff, **kwargs)

        return entries

    def serialize(self, old_snapshot, new_snapshot, diff, entity_prefix=None, **kwargs):
        entries = []
        new_entity_prefix = join_str(entity_prefix, self.entity)

        entries += self.serialize_rules(
            old_snapshot,
            new_snapshot,
            diff,
            self.rules or [],
            new_entity_prefix,
            **kwargs
        )

        for child_serializer in (self.child_serializers or []):
            entries += child_serializer.serialize(old_snapshot, new_snapshot, diff, entity_prefix=new_entity_prefix, **kwargs)

        return entries


class BaseListSerializer(object):
    """
    Этот сериализатор умеет работать только со словарями в качестве collection.
    Слово List - это обман.
    """
    entity = None

    rules = []
    list_item_entity_alias = None

    def serialize(self, old_snapshot, new_snapshot, diff, entity_prefix=None,
                  **kwargs):
        list_entity = join_str(entity_prefix, self.entity)

        list_diff = diff
        for bit in list_entity.split('.'):
            list_diff = slice_diff(list_diff, bit)
        changed_fields = list_diff.get_changed_fields()

        entries = []
        for entity_id in changed_fields:
            list_item_entity = join_str(list_entity, entity_id)
            for rule in self.rules:
                rule = rule.clone_for_prefix(list_item_entity)
                if rule.is_applicable(old_snapshot, new_snapshot, diff):
                    entries += rule.make_entries(
                        old_snapshot,
                        new_snapshot,
                        diff,
                        entity_alias=self.list_item_entity_alias,
                        entity_id=entity_id,
                        **kwargs
                    )
        # Удалим одинаковые записи, которые появляются, когда срабатывает и
        # правило удаления атрибута сущности и правило удаления сущности в
        # момент удаления аккаунта.
        entries = unique_unhashable(entries)
        return entries


class AttributeOfListItemSerializerRule(BaseSerializerRule):
    def is_applicable(self, old_snapshot, new_snapshot, diff):
        is_applicable = super(AttributeOfListItemSerializerRule, self).is_applicable(
            old_snapshot,
            new_snapshot,
            diff,
        )
        if is_applicable:
            return is_applicable

        if DELETED not in self.allowed_operations:
            return False
        if not self._list_item_deleted(diff):
            return False
        # Делать запись об удалении атрибута, нужно только если ему было задано
        # значение.
        old_value = hierarchical_get(old_snapshot, self.model_attr)
        return is_not_empty(old_value)

    def _list_item_deleted(self, diff):
        attribute_path = self.entity.split('.')
        item_entity = '.'.join(attribute_path[:-1])

        path = '%s.%s' % (DELETED, item_entity)
        if not hierarchical_exists(diff, path):
            return False
        return hierarchical_get(diff, path) is None
