# -*- coding: utf-8 -*-
from passport.backend.core.serializers.logs.base import (
    ADDED,
    AttributeOfListItemSerializerRule,
    BaseSerializerRule,
    CHANGED,
    DELETED,
)
from passport.backend.core.serializers.logs.historydb.formatters import (
    default_formatter,
    null_formatter,
)
from passport.backend.core.serializers.logs.historydb.getters import identical_getter
from passport.backend.core.serializers.logs.utils import hierarchical_get


class HistoryDBSerializerRule(BaseSerializerRule):
    """Базовое правило сериализации изменений аккаунта для HistoryDB"""
    def __init__(self, entity='', entity_alias=None,
                 operations=(ADDED, CHANGED, DELETED),
                 getter=identical_getter, formatter=default_formatter,
                 formatter_for_deleted=null_formatter):
        """
        entity: поле аккаунта, за изменением которого следим.
        entity_alias: имя поля для записи в лог. По умолчанию совпадает с entity.
        getter: функция для получения значения поля. Принимает снапшот, дифф и
                объект, лежащий по пути, указанному в entity.
        formatter: функция для конвертации значения в строку.
        """
        super(HistoryDBSerializerRule, self).__init__(
            entity=entity,
            entity_alias=entity_alias,
            operations=operations,
        )
        self.value_getter = getter
        self.value_formatter = formatter
        self.value_formatter_for_deleted = formatter_for_deleted

    def _get_values(self, old_snapshot, new_snapshot, diff, args=None):
        entity = '.'.join([self.entity] + (args or []))
        old_value = hierarchical_get(old_snapshot, entity)
        new_value = hierarchical_get(new_snapshot, entity)
        return (
            self.value_getter(old_snapshot, diff, old_value) if old_value is not None else None,
            self.value_getter(new_snapshot, diff, new_value) if new_value is not None else None,
        )

    def _get_formatted_value(self, old_snapshot, new_snapshot, diff, args=None):
        old_value, new_value = self._get_values(old_snapshot, new_snapshot, diff, args)
        if new_value is not None:
            return self.value_formatter(new_value)
        else:
            return self.value_formatter_for_deleted(old_value)

    def make_entries(self, old_snapshot, new_snapshot, diff, **kwargs):
        """Возвращает список лог-записей, созданных этим правилом"""
        formatted_value = self._get_formatted_value(old_snapshot, new_snapshot, diff)
        return [(self.entity_alias, formatted_value)]


class HistoryDBAttributeOfListItemSerializerRule(HistoryDBSerializerRule, AttributeOfListItemSerializerRule):
    pass
