# -*- coding: utf-8 -*-
from passport.backend.core.serializers.logs.historydb.formatters import default_formatter
from passport.backend.core.serializers.logs.historydb.getters import identical_getter
from passport.backend.core.serializers.logs.historydb.rules.base import (
    ADDED,
    CHANGED,
    DELETED,
    HistoryDBSerializerRule,
)
from passport.backend.core.serializers.logs.utils import (
    hierarchical_exists,
    hierarchical_get,
    is_not_empty,
)


class HistoryDBPasswordChangeSerializerRule(HistoryDBSerializerRule):
    ENTITY_FIELDS = ('encrypted_password', 'encoding_version')

    def __init__(self):
        super(HistoryDBPasswordChangeSerializerRule, self).__init__(
            operations=(ADDED, CHANGED),
        )

    def is_applicable(self, old_snapshot, new_snapshot, diff):
        """Применимо ли правило к данному диффу"""
        for field in self.ENTITY_FIELDS:
            applicable = any(
                hierarchical_exists(diff, '%s.%s.%s' % (operation, self.entity, field))
                for operation in self.allowed_operations
            )
            if applicable:
                return True
        return False

    def make_entries(self, old_snapshot, new_snapshot, diff):
        return [
            ('info.password', self.value_formatter(new_snapshot.password.serialized_encrypted_password)),
        ]


class HistoryDBPasswordDeleteSerializerRule(HistoryDBSerializerRule):
    ENTITY_ALIAS_TO_FIELD = [
        ('info.password', 'encrypted_password'),
        ('info.password_quality', 'quality'),
        ('info.password_update_time', 'update_datetime'),
    ]

    def __init__(self, entity='', entity_alias=None,
                 getter=identical_getter, formatter=default_formatter):
        super(HistoryDBPasswordDeleteSerializerRule, self).__init__(
            entity=entity,
            entity_alias=entity_alias,
            operations=(DELETED,),
            getter=getter,
        )

    def is_applicable(self, old_snapshot, new_snapshot, diff):
        old_value = hierarchical_get(old_snapshot, self.entity)
        new_value = hierarchical_get(new_snapshot, self.entity)
        return is_not_empty(old_value) and new_value is None

    def make_entries(self, old_snapshot, new_snapshot, diff):
        entries = []
        for entity_alias, _ in self.ENTITY_ALIAS_TO_FIELD:
            entries.append((entity_alias, self.value_formatter(None)))

        return entries
