# -*- coding: utf-8 -*-
from passport.backend.core.serializers.logs.base import (
    ADDED,
    CHANGED,
    DELETED,
)
from passport.backend.core.serializers.logs.historydb.formatters import sid_formatter
from passport.backend.core.serializers.logs.historydb.getters import (
    sid_add_getter,
    sid_login_rule_getter,
    sid_rm_getter,
)
from passport.backend.core.serializers.logs.historydb.rules.base import HistoryDBSerializerRule
from passport.backend.core.serializers.logs.utils import (
    hierarchical_exists,
    hierarchical_get,
)


class HistoryDBSubscriptionSerializerRule(HistoryDBSerializerRule):
    """Правило сериализации добавлений и удалений сидов"""
    def __init__(self, entity, entity_alias, operation, getter):
        super(HistoryDBSubscriptionSerializerRule, self).__init__(
            entity=entity,
            entity_alias=entity_alias,
            operations=(operation,),
            getter=getter,
            formatter=sid_formatter,
            formatter_for_deleted=sid_formatter,
        )

    def _is_applicable_for_sid(self, diff, sid):
        return True

    def make_entries(self, old_snapshot, new_snapshot, diff):
        values = []
        for operation in self.allowed_operations:
            subscriptions = hierarchical_get(diff, '%s.%s' % (operation, self.entity))
            for sid in subscriptions:
                if self._is_applicable_for_sid(diff, sid):
                    formatted_value = self._get_formatted_value(old_snapshot, new_snapshot, diff, args=[str(sid)])
                    values.append(formatted_value)
        return [(self.entity_alias, ','.join(values))] if values else []


class HistoryDBSubscriptionAddSerializerRule(HistoryDBSubscriptionSerializerRule):
    """Правило сериализации добавлений сидов"""
    def __init__(self, entity=''):
        super(HistoryDBSubscriptionAddSerializerRule, self).__init__(
            entity=entity,
            entity_alias='sid.add',
            operation=ADDED,
            getter=sid_add_getter,
        )


class HistoryDBSubscriptionLoginRuleSerializerRule(HistoryDBSubscriptionSerializerRule):
    """Правило сериализации изменений сидов (login_rule)"""
    def __init__(self, entity=''):
        super(HistoryDBSubscriptionLoginRuleSerializerRule, self).__init__(
            entity=entity,
            entity_alias='sid.login_rule',
            operation=CHANGED,
            getter=sid_login_rule_getter,
        )

    def _is_applicable_for_sid(self, diff, sid):
        return hierarchical_exists(diff, 'changed.subscriptions.%s.login_rule' % sid)


class HistoryDBSubscriptionRmSerializerRule(HistoryDBSubscriptionSerializerRule):
    """Правило сериализации удалений сидов"""
    def __init__(self, entity=''):
        super(HistoryDBSubscriptionRmSerializerRule, self).__init__(
            entity=entity,
            entity_alias='sid.rm',
            operation=DELETED,
            getter=sid_rm_getter,
        )
