# -*- coding: utf-8 -*-
from collections import defaultdict
from datetime import datetime
import logging

from passport.backend.core.historydb.entry import (
    EventEntry,
    EventTskvEntry,
)
from passport.backend.core.host.host import get_current_host
from passport.backend.core.serializers.logs.historydb.serializers import (
    HistorydbAccountSerializer,
    HistorydbFamilyInfoSerializer,
)
from passport.backend.core.serializers.runner import BaseSerializeActionRunner
from passport.backend.core.types.ip.ip import IP
from passport.backend.utils.string import always_str
from six import iteritems


event_log = logging.getLogger('historydb.event')
event_tskv_log = logging.getLogger('historydb.tskv.event')

MAX_USER_AGENT_LENGTH = 200
_CLIENT_NAME = 'passport'


class HistorydbActionRunner(BaseSerializeActionRunner):
    serializer_cls = HistorydbAccountSerializer

    def __init__(self, events, uid, user_ip, user_agent, yandexuid, host_id,
                 force_external_events=False, datetime_=None, initiator_uid=None):
        """
        Входные параметры
            force_external_events
                Вынуждает писать events в журнал, даже когда никаких других
                записей в журнал не делается.
            datetime_
                Делать все записи с данной временной меткой (объект datetime).
        """
        self.events = events
        self.user_ip = user_ip
        self.uid = uid
        self.host_id = host_id
        self.yandexuid = yandexuid
        self.force_external_events = force_external_events
        self.initiator_uid = initiator_uid

        if datetime_ is None:
            self.entry_time = datetime.now()
        else:
            self.entry_time = datetime_

        self.admin = self.events.pop('admin', '')
        self.comment = self.events.pop('comment', '')

        if user_agent is not None:
            self.events['user_agent'] = user_agent

    def get_entry(self, event_name, event_value, event_uid=None):
        event_uid = event_uid or self.uid
        entry_args = {
            'yandexuid': '',
            'admin': '',
            'comment': '',
            'user_ip': '',
        }

        is_initiator_event = self.initiator_uid == event_uid or self.initiator_uid is None

        if event_name == 'user_agent':
            if not is_initiator_event:
                return
            entry_args['value'] = event_value[:MAX_USER_AGENT_LENGTH]
        else:
            entry_args['value'] = event_value

        if is_initiator_event:
            # Пишем user_ip только тогда, когда меняем собственную учётную запись.
            entry_args['user_ip'] = self.user_ip

        if event_name == 'action':
            entry_args['admin'] = self.admin
            entry_args['comment'] = self.comment

            if is_initiator_event:
                # Пишем yandexuid только тогда, когда меняем собственную учётную
                # запись.
                entry_args['yandexuid'] = self.yandexuid
        return EventEntry(
            uid=event_uid,
            time=self.entry_time,
            name=event_name,
            client_name=_CLIENT_NAME,
            host_id=self.host_id,
            **entry_args
        )

    def _execute(self, event):
        if len(event) == 3:
            event_name, event_value, event_uid = event
        else:
            event_name, event_value = event
            event_uid = None
        event_entry = self.get_entry(event_name, event_value, event_uid)
        if event_entry is not None:
            message = str(event_entry)
            event_log.debug(message)

    def serialize(self, old_snapshot, new_snapshot, diff=None):
        is_internal_events_logged = False
        for event in self.serializer_cls().serialize(
            old_snapshot,
            new_snapshot,
            diff,
        ):
            is_internal_events_logged = True
            yield event
        if is_internal_events_logged or self.force_external_events:
            for event in iteritems(self.events):
                yield event


class HistorydbFamilyActionRunner(HistorydbActionRunner):
    serializer_cls = HistorydbFamilyInfoSerializer

    def serialize(self, old_snapshot, new_snapshot, diff=None):
        is_internal_events_logged = False
        # В отличие от обычного раннера, мы пинаем внешние события по разу
        # для каждого UID'а
        uids = set()
        for event in self.serializer_cls().serialize(
            old_snapshot,
            new_snapshot,
            diff,
        ):
            is_internal_events_logged = True
            if len(event) >= 3:
                uids.add(event[2])
            yield event
        if is_internal_events_logged or self.force_external_events:
            if uids:
                for uid in uids:
                    for event_name, event_value in iteritems(self.events):
                        yield event_name, event_value, uid
            else:
                for event in iteritems(self.events):
                    yield event


class HistorydbTskvActionRunner(BaseSerializeActionRunner):
    serializer_cls = HistorydbAccountSerializer

    def __init__(self, events, uid, user_ip, user_agent, yandexuid, host_id,
                 force_external_events=False, datetime_=None, initiator_uid=None):
        """
        Входные параметры
            force_external_events
                Вынуждает писать events в журнал, даже когда никаких других
                записей в журнал не делается.
            datetime_
                Делать все записи с данной временной меткой (объект datetime).
        """
        self.events = events
        self.ip_from = IP(user_ip).normalized
        self.uid = uid
        self.host_id = host_id
        self.yandexuid = yandexuid
        self.user_agent = user_agent[:MAX_USER_AGENT_LENGTH] if user_agent is not None else None
        self.force_external_events = force_external_events
        self.initiator_uid = initiator_uid

        if datetime_ is None:
            self.entry_time = datetime.now()
        else:
            self.entry_time = datetime_

    def get_entry(self, uid, events):
        entry_args = events.copy()
        entry_args.update(
            host_id=self.host_id,
            client_name=_CLIENT_NAME,
            uid=uid,
        )
        is_initiator_event = self.initiator_uid == uid or self.initiator_uid is None
        if is_initiator_event:
            # Ряд полей пишем только когда меняем собственную учётную запись.
            entry_args.update(
                ip_from=self.ip_from,
                yandexuid=self.yandexuid,
            )
        if self.user_agent is not None:
            entry_args.update(
                user_agent=self.user_agent
            )

        return EventTskvEntry(events=events, **entry_args)

    def _execute(self, events):
        uid, events = events
        event_entry = self.get_entry(uid=uid, events=events)
        if event_entry is not None:
            message = str(event_entry)
            message = always_str(message)
            event_tskv_log.debug(message)

    def serialize(self, old_snapshot, new_snapshot, diff=None):
        uid_to_events = defaultdict(dict)
        is_internal_events_logged = False
        for event in self.serializer_cls().serialize(
            old_snapshot,
            new_snapshot,
            diff,
        ):
            if len(event) == 3:
                event_name, event_value, uid = event
            else:
                event_name, event_value = event
                uid = self.uid

            is_internal_events_logged = True
            uid_to_events[uid][event_name] = event_value

        if is_internal_events_logged or self.force_external_events:
            for event_name, event_value in iteritems(self.events):
                uid_to_events[self.uid][event_name] = event_value
        for uid, events in uid_to_events.items():
            yield uid, events


class HistorydbTskvFamilyActionRunner(HistorydbTskvActionRunner):
    serializer_cls = HistorydbFamilyInfoSerializer

    def serialize(self, old_snapshot, new_snapshot, diff=None):
        # Для family runner внешние события (self.events) добавляем для каждого логируемого события
        # для каждого UID'а
        # Пример: {uid:123123, some_action:some_value, foo:bar, external_event_key:external_event_value }
        # Пример: {uid:321321, other_action:other_value, external_event_key:external_event_value }
        uid_to_events = defaultdict(dict)
        is_internal_events_logged = False
        for event in self.serializer_cls().serialize(
            old_snapshot,
            new_snapshot,
            diff,
        ):
            if len(event) == 3:
                event_name, event_value, uid = event
            else:
                event_name, event_value = event
                uid = self.uid

            is_internal_events_logged = True
            uid_to_events[uid][event_name] = event_value
        if is_internal_events_logged or self.force_external_events:
            for event_name, event_value in iteritems(self.events):
                for event in uid_to_events.values():
                    event[event_name] = event_value
        for uid, events in uid_to_events.items():
            yield uid, events


def log_events(user_events, user_ip, user_agent, yandexuid, initiator_uid=None, datetime_=None):
    if datetime_ is None:
        datetime_ = datetime.now()

    host_id = get_current_host().get_id()

    for uid, events in iteritems(user_events):
        if not events:
            continue

        runner = HistorydbActionRunner(
            events,
            uid,
            user_ip,
            user_agent,
            yandexuid,
            host_id,
            datetime_=datetime_,
            initiator_uid=initiator_uid,
        )
        for name, value in iteritems(runner.events):
            runner.execute((name, value))
