# -*- coding: utf-8 -*-

from passport.backend.core.differ.differ import diff as eval_diff
from passport.backend.core.differ.utils import slice_diff
from passport.backend.core.serializers.logs.base import (
    ADDED,
    BaseListSerializer,
    BaseModelSerializer,
    CHANGED,
    DELETED,
)
from passport.backend.core.serializers.logs.historydb.formatters import (
    base64_formatter,
    boolean_formatter,
    collection_formatter,
    constant_response_formatter,
    datetime_formatter,
    datetime_to_timestamp_formatter,
    default_formatter,
    disabled_to_ena_formatter,
    empty_to_null_formatter,
    json_formatter,
    login_formatter,
    mail_formatter,
    mailish_login_formatter,
    normalized_email_formatter,
    phone_e164_formatter,
    punycode_formatter,
    sid_formatter,
    zero_response_formatter,
)
from passport.backend.core.serializers.logs.historydb.getters import (
    altdomain_alias_getter,
    login_getter,
    mail_sid_info_getter,
    number_getter,
    pdd_alias_getter,
    sid_info_getter,
)
from passport.backend.core.serializers.logs.historydb.rules.aliases import (
    HistoryDBOldPublicIdsSerializerRule,
    HistoryDBPddAliasLoginsSerializerRule,
)
from passport.backend.core.serializers.logs.historydb.rules.base import (
    HistoryDBAttributeOfListItemSerializerRule,
    HistoryDBSerializerRule,
)
from passport.backend.core.serializers.logs.historydb.rules.family import (
    HistoryDBFamilyChildDeleteRule,
    HistoryDBFamilyKidDeleteRule,
)
from passport.backend.core.serializers.logs.historydb.rules.hint import (
    HistoryDBHintDeleteSerializerRule,
    HistoryDBHintSetSerializerRule,
)
from passport.backend.core.serializers.logs.historydb.rules.karma import HistoryDBKarmaSerializerRule
from passport.backend.core.serializers.logs.historydb.rules.login import HistoryDBLoginSerializerRule
from passport.backend.core.serializers.logs.historydb.rules.password import (
    HistoryDBPasswordChangeSerializerRule,
    HistoryDBPasswordDeleteSerializerRule,
)
from passport.backend.core.serializers.logs.historydb.rules.phones import (
    HistoryDBPhoneActionSerializerRule,
    HistoryDBPhoneOperationActionSerializerRule,
    HistoryDBPhoneOperationSerializerRule,
    HistoryDBPhoneSerializerRule,
)
from passport.backend.core.serializers.logs.historydb.rules.plus import HistoryDBPlusDeleteSerializerRule
from passport.backend.core.serializers.logs.historydb.rules.subscriptions import (
    HistoryDBSubscriptionAddSerializerRule,
    HistoryDBSubscriptionLoginRuleSerializerRule,
    HistoryDBSubscriptionRmSerializerRule,
)
from passport.backend.core.serializers.logs.historydb.rules.totp_secret import (
    HistoryDBTotpSecretDeleteSerializerRule,
    HistoryDBTotpSecretIdsSerializerRule,
)
from passport.backend.core.undefined import Undefined


class HistorydbPersonSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule('firstname', entity_alias='info.firstname'),
        HistoryDBSerializerRule('lastname', entity_alias='info.lastname'),
        HistoryDBSerializerRule('display_name', entity_alias='info.display_name'),
        HistoryDBSerializerRule('gender', entity_alias='info.sex'),
        HistoryDBSerializerRule('birthday', entity_alias='info.birthday'),
        HistoryDBSerializerRule('city', entity_alias='info.city'),
        HistoryDBSerializerRule('country', entity_alias='info.country'),
        HistoryDBSerializerRule('email', entity_alias='info.email'),
        HistoryDBSerializerRule('timezone', entity_alias='info.tz'),
        HistoryDBSerializerRule('language', entity_alias='info.lang'),
        HistoryDBSerializerRule('default_avatar', entity_alias='info.default_avatar'),
        HistoryDBSerializerRule(
            'dont_use_displayname_as_public_name',
            entity_alias='info.dont_use_displayname_as_public_name',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'show_fio_in_public_name',
            entity_alias='info.show_fio_in_public_name',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule('firstname_global', entity_alias='info.firstname_global'),
        HistoryDBSerializerRule('lastname_global', entity_alias='info.lastname_global'),
    )


class HistorydbPlusSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'enabled',
            entity_alias='plus.enabled',
            formatter=boolean_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBSerializerRule(
            'trial_used_ts',
            entity_alias='plus.trial_used_ts',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBSerializerRule(
            'subscription_stopped_ts',
            entity_alias='plus.subscription_stopped_ts',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBSerializerRule(
            'subscription_expire_ts',
            entity_alias='plus.subscription_expire_ts',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBSerializerRule(
            'next_charge_ts',
            entity_alias='plus.next_charge_ts',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBSerializerRule(
            'ott_subscription',
            entity_alias='plus.ott_subscription',
            formatter=empty_to_null_formatter,
        ),
        HistoryDBSerializerRule(
            'family_role',
            entity_alias='plus.family_role',
            formatter=empty_to_null_formatter,
        ),
        HistoryDBSerializerRule(
            'cashback_enabled',
            entity_alias='plus.cashback_enabled',
            formatter=boolean_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBSerializerRule(
            'subscription_level',
            entity_alias='plus.subscription_level',
            formatter=default_formatter,
        ),
        HistoryDBSerializerRule(
            'is_frozen',
            entity_alias='plus.is_frozen',
            formatter=boolean_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBSerializerRule(
            'subscriber_state',
            entity_alias='plus.subscriber_state',
            formatter=empty_to_null_formatter,
        ),
        HistoryDBPlusDeleteSerializerRule(),
    )


class HistorydbTakeoutSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'extract_in_progress_since',
            entity_alias='takeout.extract_in_progress_since',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBSerializerRule(
            'archive_s3_key',
            entity_alias='takeout.archive_s3_key',
            formatter=constant_response_formatter('***'),
        ),
        HistoryDBSerializerRule(
            'archive_password',
            entity_alias='takeout.archive_password',
            formatter=constant_response_formatter('***'),
        ),
        HistoryDBSerializerRule(
            'archive_created_at',
            entity_alias='takeout.archive_created_at',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBSerializerRule(
            'fail_extract_at',
            entity_alias='takeout.fail_extract_at',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBSerializerRule(
            'subscription',
            entity_alias='takeout.subscription',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'delete_subscription',
            entity_alias='takeout.delete.subscription',
            formatter=boolean_formatter,
        ),
    )


class HistorydbPasswordSerializer(BaseModelSerializer):
    rules = (
        HistoryDBPasswordChangeSerializerRule(),
        HistoryDBSerializerRule(
            'quality',
            entity_alias='info.password_quality',
            operations=(ADDED, CHANGED,),
        ),
        HistoryDBSerializerRule(
            'update_datetime',
            entity_alias='info.password_update_time',
            formatter=datetime_to_timestamp_formatter,
            operations=(ADDED, CHANGED,),
        ),
        HistoryDBSerializerRule(
            'pwn_forced_changing_suspended_at',
            entity_alias='info.password_pwn_forced_changing_suspension_time',
            formatter=datetime_to_timestamp_formatter,
            operations=(ADDED, CHANGED, DELETED),
        ),

        HistoryDBPasswordDeleteSerializerRule(),
    )


class HistorydbScholarPasswordSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'serialized_encrypted_password',
            entity_alias='info.scholar_password',
        ),
    )


class HistorydbRfcTotpSecretSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'totp_secret',
            entity_alias='info.rfc_totp',
            formatter=constant_response_formatter('enabled'),
            operations=(ADDED,),
        ),

        HistoryDBSerializerRule(
            entity_alias='info.rfc_totp',
            operations=(DELETED,),
            formatter_for_deleted=constant_response_formatter('disabled'),
        ),
    )


class HistorydbTotpSecretSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'update_datetime',
            entity_alias='info.totp_update_time',
            formatter=datetime_to_timestamp_formatter,
            operations=(ADDED,),
        ),
        HistoryDBTotpSecretIdsSerializerRule(),
        HistoryDBSerializerRule(
            'yakey_device_ids',
            entity_alias='info.totp_yakey_device_ids',
            formatter=collection_formatter,
            operations=(ADDED, CHANGED),
        ),

        HistoryDBTotpSecretDeleteSerializerRule(),
    )


class HistorydbPddDomainSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'domain',
            entity_alias='info.domain_name',
            operations=(ADDED, CHANGED, DELETED),
            formatter=punycode_formatter,
        ),
        HistoryDBSerializerRule(
            'id',
            entity_alias='info.domain_id',
            operations=(ADDED, CHANGED, DELETED),
        ),
    )


class HistorydbHintSerializer(BaseModelSerializer):
    rules = (
        HistoryDBHintSetSerializerRule(),
        HistoryDBHintDeleteSerializerRule(),
    )


class HistorydbKarmaSerializer(BaseModelSerializer):
    rules = (
        HistoryDBKarmaSerializerRule('prefix', entity_alias='info.karma_prefix'),
        HistoryDBKarmaSerializerRule('suffix', entity_alias='info.karma'),
    )


class HistorydbPortalAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'login',
            entity_alias='alias.portal.add',
            operations=(ADDED,),
            formatter=login_formatter,
        ),
    )


class HistorydbPddAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'email',
            entity_alias='alias.pdd.add',
            getter=pdd_alias_getter,
            operations=(ADDED,),
        ),
        HistoryDBPddAliasLoginsSerializerRule(
            'additional_logins',
            operations=(ADDED, CHANGED, DELETED),
        ),
    )


class HistorydbPublicIdAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'alias',
            entity_alias='alias.public_id.add',
            operations=(ADDED,),
        ),
        HistoryDBSerializerRule(
            'alias',
            entity_alias='alias.public_id.upd',
            operations=(CHANGED,),
        ),
        HistoryDBSerializerRule(
            'alias',
            entity_alias='alias.public_id.rm',
            operations=(DELETED,),
            formatter_for_deleted=default_formatter,
        ),
        HistoryDBOldPublicIdsSerializerRule(
            'old_public_ids',
            operations=(ADDED, CHANGED, DELETED),
        ),
    )


class HistorydbLiteAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'email',
            entity_alias='alias.lite.add',
            operations=(ADDED,),
        ),
    )


class HistorydbScholarAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'alias',
            entity_alias='alias.scholar.add',
            operations=(ADDED,),
        ),
    )


class HistorydbSocialAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'login',
            entity_alias='alias.social.add',
            operations=(ADDED,),
        ),
    )


class HistorydbPhonishAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'login',
            entity_alias='alias.phonish.add',
            operations=(ADDED,),
        ),
    )


class HistorydbNeophonishAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'alias',
            entity_alias='alias.neophonish.add',
            operations=(ADDED,),
        ),
    )


class HistorydbMailishAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'mailish_id',
            entity_alias='alias.mailish.add',
            operations=(ADDED,),
            formatter=mailish_login_formatter,
        ),
        HistoryDBSerializerRule(
            'mailish_id',
            entity_alias='alias.mailish.change',
            operations=(CHANGED,),
            formatter=mailish_login_formatter,
        ),
    )


class HistorydbKiddishAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'alias',
            entity_alias='alias.kiddish.add',
            operations=(ADDED,),
        ),
    )


class HistorydbKinopoiskAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'alias',
            entity_alias='alias.kinopoisk.add',
            operations=(ADDED,),
        ),
    )


class HistorydbUberAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'uber_id',
            entity_alias='alias.uber.add',
            operations=(ADDED,),
        ),
    )


class HistorydbYambotAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'alias',
            entity_alias='alias.yambot.add',
            operations=(ADDED,),
        ),
    )


class HistorydbKolonkishAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'alias',
            entity_alias='alias.kolonkish.add',
            operations=(ADDED,),
        ),
    )


class _HistorydbPhonenumberAliasChangeSerializerRule(HistoryDBSerializerRule):
    def __init__(self):
        super(_HistorydbPhonenumberAliasChangeSerializerRule, self).__init__(
            'number',
            operations=(CHANGED,),
        )

    def make_entries(self, old_snapshot, new_snapshot, diff):
        return [
            ('alias.phonenumber.change', self.value_formatter(new_snapshot.phonenumber_alias.number)),
        ]


class HistorydbPhonenumberAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'number',
            entity_alias='alias.phonenumber.add',
            operations=(ADDED,),
        ),
        HistoryDBSerializerRule(
            entity_alias='alias.phonenumber.rm',
            operations=(DELETED,),
            getter=number_getter,
            formatter_for_deleted=default_formatter,
        ),
        _HistorydbPhonenumberAliasChangeSerializerRule(),
        HistoryDBSerializerRule(
            'enable_search',
            entity_alias='info.phonenumber_alias_search_enabled',
            formatter=boolean_formatter,
        ),
    )


class HistorydbAltDomainAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'login',
            entity_alias='alias.altdomain.add',
            operations=(ADDED,),
            getter=altdomain_alias_getter,
        ),
        HistoryDBSerializerRule(
            entity_alias='alias.altdomain.rm',
            operations=(DELETED,),
            getter=altdomain_alias_getter,
            formatter_for_deleted=default_formatter,
        ),
    )


class HistorydbYandexoidAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'login',
            entity_alias='alias.yandexoid.add',
            operations=(ADDED,),
            formatter=login_formatter,
        ),
        HistoryDBSerializerRule(
            entity_alias='alias.yandexoid.rm',
            operations=(DELETED,),
            getter=login_getter,
            formatter_for_deleted=default_formatter,
        ),
    )


class HistorydbBankPhoneNumberAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'alias',
            entity_alias='alias.bank_phonenumber.add',
            operations=(ADDED,),
        ),
        HistoryDBSerializerRule(
            entity_alias='alias.bank_phonenumber.rm',
            operations=(DELETED,),
        ),
    )


class HistorydbFederalAliasSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'login',
            entity_alias='alias.federal.add',
            operations=(ADDED,),
        ),
    )


class HistorydbSubscriptionsSerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            '42.host.id',
            entity_alias='sid.wwwdgt_wmode',
            operations=(ADDED, CHANGED),
        ),
        HistoryDBSerializerRule(
            '2',
            entity_alias='sid.rm.info',
            operations=(DELETED,),
            getter=sid_info_getter,
            formatter_for_deleted=sid_formatter,
        ),

        HistoryDBSerializerRule(
            '2',
            entity_alias='mail.add',
            operations=(ADDED,),
            getter=mail_sid_info_getter,
            formatter=mail_formatter,
        ),
        HistoryDBSerializerRule(
            '2',
            entity_alias='mail.rm',
            operations=(DELETED,),
            getter=mail_sid_info_getter,
            formatter_for_deleted=mail_formatter,
        ),

        HistoryDBSubscriptionAddSerializerRule(''),
        HistoryDBSubscriptionLoginRuleSerializerRule(''),
        HistoryDBSubscriptionRmSerializerRule(''),
    )


class HistorydbPhoneOperationSerializer(BaseModelSerializer):
    basic_rules = (
        HistoryDBPhoneOperationSerializerRule(
            'type',
            formatter_for_deleted=default_formatter,
        ),
        HistoryDBPhoneOperationActionSerializerRule('action'),
        HistoryDBPhoneOperationSerializerRule(
            'security_identity',
            formatter_for_deleted=default_formatter,
        ),
    )

    rules = (
        HistoryDBPhoneOperationSerializerRule(
            'started',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBPhoneOperationSerializerRule(
            'finished',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBPhoneOperationSerializerRule(
            'code_confirmed',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBPhoneOperationSerializerRule(
            'password_verified',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBPhoneOperationSerializerRule(
            'phone_id2',
            formatter=default_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
    )

    def serialize(self, old_phone, new_phone, diff, phone_id):
        new_operation = new_phone.operation if new_phone else None
        old_operation = old_phone.operation if old_phone else None
        entries = []

        if (new_operation and
            old_operation and
                new_operation.id == old_operation.id):
            # изменение
            entries += self.serialize_rules(
                old_phone,
                new_phone,
                diff,
                self.rules,
                self.entity,
                operation_id=new_operation.id,
                phone_id=phone_id,
            )

            if entries:
                entries += self.serialize_rules(
                    old_phone,
                    new_phone,
                    diff,
                    self.basic_rules,
                    self.entity,
                    check_is_applicable=False,
                    operation_id=new_operation.id,
                    phone_id=phone_id,
                )
            return entries

        if old_operation:
            # удаление старой операции
            entries += self.serialize_rules(
                old_phone,
                None,
                eval_diff(old_phone, None),
                self.basic_rules,
                self.entity,
                check_is_applicable=False,
                operation_id=old_operation.id,
                phone_id=phone_id,
            )

        if new_operation:
            # создание новой операции
            entries += self.serialize_rules(
                None,
                new_phone,
                eval_diff(None, new_phone),
                self.rules,
                self.entity,
                operation_id=new_operation.id,
                phone_id=phone_id,
            )
            entries += self.serialize_rules(
                None,
                new_phone,
                eval_diff(None, new_phone),
                self.basic_rules,
                self.entity,
                check_is_applicable=False,
                operation_id=new_operation.id,
                phone_id=phone_id,
            )

        return entries


class HistorydbPhoneSerializer(BaseModelSerializer):
    rule_number = HistoryDBPhoneSerializerRule(
        'number',
        formatter=phone_e164_formatter,
        formatter_for_deleted=phone_e164_formatter,
    )

    # Если поменялись только поля операции, то не пишем action, но пишем number
    rule_action = HistoryDBPhoneActionSerializerRule('action')

    rules = (
        HistoryDBPhoneSerializerRule(
            'created',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBPhoneSerializerRule(
            '_bound',
            entity_alias='bound',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBPhoneSerializerRule(
            'confirmed',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBPhoneSerializerRule(
            'admitted',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBPhoneSerializerRule(
            'secured',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
    )

    def serialize(self, old_phone, new_phone, diff):
        phone_id = old_phone.id if old_phone else new_phone.id
        phone_entries = []

        if new_phone:
            phone_entries += self.serialize_rules(
                old_phone,
                new_phone,
                diff,
                self.rules,
                self.entity,
                phone_id=phone_id,
            )

        is_phone_created = not old_phone and new_phone
        is_phone_deleted = old_phone and not new_phone
        is_number_changed = old_phone and new_phone and old_phone.number != new_phone.number

        if is_phone_created or is_phone_deleted or phone_entries or is_number_changed:
            phone_entries += self.serialize_rules(
                old_phone,
                new_phone,
                diff,
                [self.rule_action],
                self.entity,
                check_is_applicable=False,
                phone_id=phone_id,
            )

        op_entries = HistorydbPhoneOperationSerializer('operation').serialize(
            old_phone,
            new_phone,
            diff,
            phone_id=phone_id,
        )

        # Номер нужен, только когда есть хотя бы одна запись о телефоне или об операции.
        if phone_entries or op_entries:
            phone_entries += self.serialize_rules(
                old_phone,
                new_phone,
                diff,
                [self.rule_number],
                self.entity,
                check_is_applicable=False,
                phone_id=phone_id,
            )

        return phone_entries + op_entries


class HistorydbPhonesSerializer(BaseModelSerializer):
    rule_default = HistoryDBSerializerRule(
        'default_id',
        entity_alias='phones.default',
        formatter_for_deleted=zero_response_formatter,
    )

    rule_secure = HistoryDBSerializerRule(
        'secure_id',
        entity_alias='phones.secure',
        formatter_for_deleted=zero_response_formatter,
    )

    rules = (
        rule_default,
        rule_secure,
    )

    def _is_rule_applicable(self, rule, old_snapshot, new_snapshot, phones_diff):
        """
        Проверяем атрибуты default и secure. Возвращаем phone_id, попавшие в лог.
        """
        return rule.is_applicable(
            old_snapshot.phones if old_snapshot else None,
            new_snapshot.phones if new_snapshot else None,
            phones_diff,
        )

    def _get_default_phone(self, account):
        if account:
            default_id = account.phones.default_id
            if default_id:
                return account.phones.by_id(default_id)

    def _get_secure_phone(self, account):
        if account:
            secure_id = account.phones.secure_id
            if secure_id:
                return account.phones.by_id(secure_id)

    def serialize(self, old_snapshot, new_snapshot, diff, entity_prefix=None, **kwargs):
        entries = super(HistorydbPhonesSerializer, self).serialize(
            old_snapshot,
            new_snapshot,
            diff,
            **kwargs
        )

        phones_diff = slice_diff(diff, 'phones')
        internal_phones_diff = slice_diff(phones_diff, '_phones')

        # Если изменились default или secure - напишем еще и номер телефона

        if self._is_rule_applicable(self.rule_default, old_snapshot, new_snapshot, phones_diff):
            old_default = self._get_default_phone(old_snapshot)
            new_default = self._get_default_phone(new_snapshot)

            if new_default:
                phone_id = new_default.id
            else:
                phone_id = old_default.id
            entries += HistorydbPhoneSerializer.rule_number.make_entries(
                old_default,
                new_default,
                slice_diff(internal_phones_diff, phone_id),
                phone_id=phone_id,
            )

        if self._is_rule_applicable(self.rule_secure, old_snapshot, new_snapshot, phones_diff):
            old_secure = self._get_secure_phone(old_snapshot)
            new_secure = self._get_secure_phone(new_snapshot)

            if new_secure:
                phone_id = new_secure.id
            else:
                phone_id = old_secure.id
            entries += HistorydbPhoneSerializer.rule_number.make_entries(
                old_secure,
                new_secure,
                slice_diff(internal_phones_diff, phone_id),
                phone_id=phone_id,
            )

        # Пройдемся по всем изменённым телефонам
        for phone_id in internal_phones_diff.get_changed_fields():
            if old_snapshot:
                old_phone = old_snapshot.phones.by_id(phone_id, assert_exists=False)
            else:
                old_phone = None

            entries += HistorydbPhoneSerializer().serialize(
                old_phone,
                new_snapshot.phones.by_id(phone_id, assert_exists=False) if new_snapshot else None,
                slice_diff(internal_phones_diff, phone_id),
            )

        # Удалим дублирующиеся ключи. Такими ключами могут быть phone.id.number
        entries = list(dict(entries).items())
        entries.sort()

        return entries


class HistorydbEmailSerializer(BaseModelSerializer):
    rules = [
        HistoryDBSerializerRule(
            'confirmed_at',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBSerializerRule(
            'created_at',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBSerializerRule(
            'bound_at',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
        HistoryDBSerializerRule(
            'is_silent',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'is_rpop',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'is_native',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'is_default',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'is_unsafe',
            formatter=boolean_formatter,
        ),
    ]


class HistorydbEmailsSerializer(BaseModelSerializer):
    def serialize(self, old_snapshot, new_snapshot, diff, entity_prefix=None, **kwargs):
        entries = super(HistorydbEmailsSerializer, self).serialize(
            old_snapshot,
            new_snapshot,
            diff,
            **kwargs
        )

        emails_diff = slice_diff(diff, 'emails')
        if emails_diff.deleted is None:
            temporary_entries = []
            # В данном случае нас интересуют только внешние адреса
            # с установленным ID (есть при условии жизни адреса
            # в новой базе и запросе email_attributes от ЧЯ)
            for email in old_snapshot.emails.external:
                if email.id is Undefined:
                    continue

                temporary_entries.extend([
                    ('email.%d' % email.id, 'deleted'),
                    (
                        'email.%d.address' % email.id,
                        normalized_email_formatter(email.address),
                    ),
                ])
            entries.extend(temporary_entries)
            return entries

        internal_emails_diff = slice_diff(emails_diff, '_emails')
        for address in internal_emails_diff.get_changed_fields():
            old_email = new_email = None
            temporary_entries = []

            if old_snapshot:
                old_email = old_snapshot.emails.find(address)
            if new_snapshot and new_snapshot.emails:
                new_email = new_snapshot.emails.find(address)

            if old_email or new_email:
                temporary_entries += HistorydbEmailSerializer().serialize(
                    old_email,
                    new_email,
                    slice_diff(internal_emails_diff, address),
                    **kwargs
                )
                instance = new_email or old_email

                if instance.id is Undefined:
                    continue

                if not new_email:
                    action = 'deleted'
                elif not old_email:
                    action = 'created'
                else:
                    action = 'updated'

                entries.extend([
                    ('email.%d' % instance.id, action),
                    (
                        'email.%d.address' % instance.id,
                        normalized_email_formatter(instance.address),
                    ),
                ])
                # Прибавляем ID измененного Email'а
                entries.extend([
                    ('email.%d.%s' % (instance.id, k), v)
                    for k, v in temporary_entries
                ])

        return entries


class HistorydbWebauthnCredentialSerializer(BaseModelSerializer):
    rules = [
        HistoryDBSerializerRule(
            'public_key',
        ),
        HistoryDBSerializerRule(
            'relying_party_id',
        ),
        HistoryDBSerializerRule(
            'sign_count',
        ),
        HistoryDBSerializerRule(
            'device_name',
        ),
        HistoryDBSerializerRule(
            'os_family_id',
        ),
        HistoryDBSerializerRule(
            'browser_id',
        ),
        HistoryDBSerializerRule(
            'is_device_mobile',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'is_device_tablet',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'created_at',
            formatter=datetime_to_timestamp_formatter,
            formatter_for_deleted=zero_response_formatter,
        ),
    ]


class HistorydbWebauthnCredentialsSerializer(BaseModelSerializer):
    def serialize(self, old_snapshot, new_snapshot, diff, entity_prefix=None, **kwargs):
        entries = super(HistorydbWebauthnCredentialsSerializer, self).serialize(
            old_snapshot,
            new_snapshot,
            diff,
            **kwargs
        )

        creds_diff = slice_diff(diff, 'webauthn_credentials')
        if creds_diff.deleted is None:
            # Модель WebauthnCredentials удалили целиком
            for cred in old_snapshot.webauthn_credentials.all():
                entries.extend([
                    ('webauthn_cred.%d' % cred.id, 'deleted'),
                    ('webauthn_cred.%d.external_id' % cred.id, cred.external_id),
                ])
            return entries

        creds_diff_internal = slice_diff(creds_diff, '_creds_by_cred_id')
        for cred_external_id in creds_diff_internal.get_changed_fields():
            old_cred = new_cred = None
            temporary_entries = []

            if old_snapshot and old_snapshot.webauthn_credentials:
                old_cred = old_snapshot.webauthn_credentials.by_external_id(cred_external_id)
            if new_snapshot and new_snapshot.webauthn_credentials:
                new_cred = new_snapshot.webauthn_credentials.by_external_id(cred_external_id)

            if old_cred or new_cred:
                temporary_entries += HistorydbWebauthnCredentialSerializer().serialize(
                    old_cred,
                    new_cred,
                    slice_diff(creds_diff_internal, cred_external_id),
                    **kwargs
                )
                instance = old_cred or new_cred

                if instance.id is Undefined:
                    continue

                if not new_cred:
                    action = 'deleted'
                elif not old_cred:
                    action = 'created'
                else:
                    action = 'updated'

                entries.extend([
                    ('webauthn_cred.%d' % instance.id, action),
                    ('webauthn_cred.%d.external_id' % instance.id, instance.external_id),
                ])
                entries.extend([
                    ('webauthn_cred.%d.%s' % (instance.id, k), v)
                    for k, v in temporary_entries
                ])

        return entries


class HistorydbPassManRecoveryKeySerializer(BaseModelSerializer):
    rules = (
        HistoryDBSerializerRule(
            'key_id',
            entity_alias='info.passman_key_id',
            operations=(ADDED,),
            formatter=base64_formatter,
        ),
    )


class HistorydbAccountDeletionOperationSerializer(BaseModelSerializer):
    def serialize(self, old_snapshot, new_snapshot, diff, **kwargs):
        entries = super(HistorydbAccountDeletionOperationSerializer, self).serialize(old_snapshot, new_snapshot, diff, **kwargs)

        old_op = old_snapshot and old_snapshot.deletion_operation
        new_op = new_snapshot and new_snapshot.deletion_operation

        if new_op and not old_op:
            entries += [('deletion_operation', 'created')]
        elif new_op and old_op:
            update_entries = []
            if new_op.started_at != old_op.started_at:
                update_entries += [('deletion_operation.started_at', datetime_formatter(new_op.started_at))]
            if update_entries:
                entries += [('deletion_operation', 'updated')] + update_entries
        elif not new_op and old_op:
            entries += [('deletion_operation', 'deleted')]

        return entries


class HistorydbAccountSerializer(BaseModelSerializer):
    rules = (
        HistoryDBLoginSerializerRule(),
        HistoryDBSerializerRule(
            'disabled_status',
            entity_alias='info.ena',
            formatter=disabled_to_ena_formatter,
        ),
        HistoryDBSerializerRule(
            'disabled_status',
            entity_alias='info.disabled_status',
        ),
        HistoryDBSerializerRule(
            'global_logout_datetime',
            entity_alias='info.glogout',
            formatter=datetime_to_timestamp_formatter,
        ),
        HistoryDBSerializerRule(
            'tokens_revoked_at',
            entity_alias='info.tokens_revoked',
            formatter=datetime_to_timestamp_formatter,
        ),
        HistoryDBSerializerRule(
            'web_sessions_revoked_at',
            entity_alias='info.web_sessions_revoked',
            formatter=datetime_to_timestamp_formatter,
        ),
        HistoryDBSerializerRule(
            'app_passwords_revoked_at',
            entity_alias='info.app_passwords_revoked',
            formatter=datetime_to_timestamp_formatter,
        ),
        HistoryDBSerializerRule(
            'registration_datetime',
            entity_alias='info.reg_date',
            formatter=datetime_formatter,
        ),
        HistoryDBSerializerRule(
            'is_employee',
            entity_alias='info.is_employee',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'is_maillist',
            entity_alias='info.is_maillist',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'default_email',
            entity_alias='info.default_email',
        ),
        HistoryDBSerializerRule(
            'enable_app_password',
            entity_alias='info.enable_app_password',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'is_shared',
            entity_alias='info.is_shared',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'is_easily_hacked',
            entity_alias='info.is_easily_hacked',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'is_connect_admin',
            entity_alias='info.is_connect_admin',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'audience_on',
            entity_alias='info.audience_on',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'is_money_agreement_accepted',
            entity_alias='info.money_eula_accepted',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'additional_data_asked',
            entity_alias='info.additional_data_asked',
        ),
        HistoryDBSerializerRule(
            'additional_data_ask_next_datetime',
            entity_alias='info.additional_data_ask_next_datetime',
            formatter=datetime_formatter,
        ),
        HistoryDBSerializerRule(
            'content_rating_class',
            entity_alias='info.content_rating_class',
        ),
        HistoryDBSerializerRule(
            'creator_uid',
            entity_alias='account.creator_uid',
        ),
        HistoryDBSerializerRule(
            'external_organization_ids',
            entity_alias='info.external_organization_ids',
            formatter=collection_formatter,
        ),
        HistoryDBSerializerRule(
            'music_content_rating_class',
            entity_alias='info.music_content_rating_class',
        ),
        HistoryDBSerializerRule(
            'phonish_namespace',
            entity_alias='info.phonish_namespace',
        ),
        HistoryDBSerializerRule(
            'video_content_rating_class',
            entity_alias='info.video_content_rating_class',
        ),
        HistoryDBSerializerRule(
            'magic_link_login_forbidden',
            entity_alias='info.magic_link_login_forbidden',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'qr_code_login_forbidden',
            entity_alias='info.qr_code_login_forbidden',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'sms_code_login_forbidden',
            entity_alias='info.sms_code_login_forbidden',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'billing_features',
            entity_alias='account.billing_features',
            formatter=json_formatter,
        ),
        HistoryDBSerializerRule(
            'force_challenge',
            entity_alias='account.force_challenge',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'user_defined_public_id',
            entity_alias='account.user_defined_public_id',
        ),
        HistoryDBSerializerRule(
            'sms_2fa_on',
            entity_alias='info.sms_2fa_on',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'forbid_disabling_sms_2fa',
            entity_alias='info.forbid_disabling_sms_2fa',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'is_verified',
            entity_alias='info.is_verified',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'hide_yandex_domains_emails',
            entity_alias='info.hide_yandex_domains_emails',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'mail_status',
            entity_alias='info.mail_status',
        ),
        HistoryDBFamilyChildDeleteRule(),
        HistoryDBFamilyKidDeleteRule(),
        HistoryDBSerializerRule(
            'unsubscribed_from_maillists',
            entity_alias='account.unsubscribed_from_maillists',
        ),
        HistoryDBSerializerRule(
            'personal_data_public_access_allowed',
            entity_alias='account.personal_data_public_access_allowed',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'personal_data_third_party_processing_allowed',
            entity_alias='account.personal_data_third_party_processing_allowed',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'family_pay_enabled',
            entity_alias='account.family_pay.enabled',
        ),
        HistoryDBSerializerRule(
            'is_documents_agreement_accepted',
            entity_alias='account.is_documents_agreement_accepted',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'is_dzen_sso_prohibited',
            entity_alias='account.is_dzen_sso_prohibited',
            formatter=boolean_formatter,
        ),
        HistoryDBSerializerRule(
            'last_child_family',
            entity_alias='account.last_child_family',
        ),
        HistoryDBSerializerRule(
            'can_manage_children',
            entity_alias='account.can_manage_children',
            formatter=boolean_formatter,
        )
    )
    child_serializers = (
        HistorydbPersonSerializer('person'),
        HistorydbPlusSerializer('plus'),
        HistorydbTakeoutSerializer('takeout'),
        HistorydbPddDomainSerializer('domain'),
        HistorydbPasswordSerializer('password'),
        HistorydbTotpSecretSerializer('totp_secret'),
        HistorydbRfcTotpSecretSerializer('rfc_totp_secret'),
        HistorydbHintSerializer('hint'),
        HistorydbKarmaSerializer('karma'),
        HistorydbPortalAliasSerializer('portal_alias'),
        HistorydbPddAliasSerializer('pdd_alias'),
        HistorydbLiteAliasSerializer('lite_alias'),
        HistorydbScholarAliasSerializer('scholar_alias'),
        HistorydbSocialAliasSerializer('social_alias'),
        HistorydbPhonishAliasSerializer('phonish_alias'),
        HistorydbNeophonishAliasSerializer('neophonish_alias'),
        HistorydbMailishAliasSerializer('mailish_alias'),
        HistorydbKinopoiskAliasSerializer('kinopoisk_alias'),
        HistorydbUberAliasSerializer('uber_alias'),
        HistorydbYambotAliasSerializer('yambot_alias'),
        HistorydbKiddishAliasSerializer('kiddish_alias'),
        HistorydbKolonkishAliasSerializer('kolonkish_alias'),
        HistorydbPublicIdAliasSerializer('public_id_alias'),
        HistorydbPhonenumberAliasSerializer('phonenumber_alias'),
        HistorydbAltDomainAliasSerializer('altdomain_alias'),
        HistorydbYandexoidAliasSerializer('yandexoid_alias'),
        HistorydbBankPhoneNumberAliasSerializer('bank_phonenumber_alias'),
        HistorydbFederalAliasSerializer('federal_alias'),
        HistorydbSubscriptionsSerializer('subscriptions'),
        HistorydbPhonesSerializer('phones'),
        HistorydbEmailsSerializer('emails'),
        HistorydbWebauthnCredentialsSerializer('webauthn_credentials'),
        HistorydbPassManRecoveryKeySerializer('passman_recovery_key'),
        HistorydbAccountDeletionOperationSerializer('deletion_operation'),
        HistorydbScholarPasswordSerializer('scholar_password'),
    )


class HistoryDBFamilyAdminUidSerializerRule(HistoryDBSerializerRule):
    def make_entries(self, old_snapshot, new_snapshot, diff, **kwargs):
        formatted_value = self._get_formatted_value(old_snapshot, new_snapshot, diff)
        snapshot = new_snapshot or old_snapshot
        return [(
            'family.%s.%s' % (snapshot.family_id, self.entity_alias),
            formatted_value,
            str(snapshot.admin_uid),
        )]


class HistoryDBFamilyMemberUidSerializerRule(HistoryDBAttributeOfListItemSerializerRule):
    def make_entries(self, old_snapshot, new_snapshot, diff, **kwargs):
        formatted_value = self._get_formatted_value(old_snapshot, new_snapshot, diff)
        snapshot = new_snapshot or old_snapshot
        uid = self.entity.split('.')[-2]
        return [(
            'family.%s.%s' % (snapshot.family_id, self.entity_alias),
            formatted_value,
            uid,
        )]


class HistorydbFamilyMembersSerializer(BaseListSerializer):
    entity = 'members'

    rules = (
        HistoryDBFamilyMemberUidSerializerRule('uid', 'family_member'),
    )


class HistorydbFamilyKidsSerializer(BaseListSerializer):
    entity = 'kids.members'

    rules = (
        HistoryDBFamilyMemberUidSerializerRule('uid', 'family_kid'),
    )


class HistorydbFamilyInfoSerializer(BaseModelSerializer):
    rules = (
        HistoryDBFamilyAdminUidSerializerRule('admin_uid', 'family_admin'),
    )

    child_serializers = (
        HistorydbFamilyMembersSerializer(),
        HistorydbFamilyKidsSerializer(),
    )
