# -*- coding: utf-8 -*-
from passport.backend.core.eav_type_mapping import ALIAS_NAME_TO_TYPE
from passport.backend.core.serializers.logs.base import (
    ADDED,
    CHANGED,
    DELETED,
)
from passport.backend.core.serializers.logs.statbox.formatters import default_formatter
from passport.backend.core.serializers.logs.statbox.rules.base import StatboxSerializerRule
from passport.backend.core.services import ALT_DOMAIN_ID_TO_SID
from passport.backend.core.undefined import Undefined


class StatboxAliasAddSerializerRule(StatboxSerializerRule):
    """Правило сериализации создания алиасов (логируем соответствующие им сиды)"""
    def __init__(self, entity, sid):
        super(StatboxAliasAddSerializerRule, self).__init__(
            entity=entity,
            entity_alias='subscriptions',
            operations=(ADDED,),
            getter=None,
            formatter=None,
        )
        self.sid = sid

    def make_entries(self, old_snapshot, new_snapshot, diff, action):
        return [self._make_entry(old_snapshot, new_snapshot, diff, action, 'added', sid=self.sid)]


class StatboxAliasRmSerializerRule(StatboxSerializerRule):
    """Правило сериализации удаления алиасов (логируем соответствующие им сиды)"""
    def __init__(self, sid):
        super(StatboxAliasRmSerializerRule, self).__init__(
            entity='',
            entity_alias='subscriptions',
            operations=(DELETED,),
            getter=None,
            formatter=None,
        )
        self.sid = sid

    def make_entries(self, old_snapshot, new_snapshot, diff, action):
        return [
            self._make_entry(
                old_snapshot,
                new_snapshot,
                diff,
                action,
                'removed',
                sid=self.sid,
            ),
        ]


class StatboxPddAliasAdditionalLoginsSerializerRule(StatboxSerializerRule):
    def __init__(self):
        super(StatboxPddAliasAdditionalLoginsSerializerRule, self).__init__(
            entity='pdd_alias_login',
            operations=(ADDED, CHANGED, DELETED,),
            getter=None,
            formatter=None,
        )

    def make_entries(self, old_snapshot, new_snapshot, diff, action):
        old_logins = old_snapshot.pdd_alias.additional_logins if old_snapshot else set()
        new_logins = new_snapshot.pdd_alias.additional_logins if new_snapshot else set()

        added_logins = new_logins - old_logins
        removed_logins = old_logins - new_logins

        entries = []
        for operation, logins in (
            ('added', sorted(added_logins)),
            ('removed', sorted(removed_logins)),
        ):
            for login in logins:
                entries.append(
                    self._make_entry(
                        old_snapshot,
                        new_snapshot,
                        diff,
                        action,
                        operation,
                        type=ALIAS_NAME_TO_TYPE['pddalias'],
                        login=login,
                    ),
                )
        return entries

    def is_applicable(self, old_snapshot, new_snapshot, diff):
        return 'pdd_alias' in diff.changed


class StatboxPublicIdAliasOldPublicIdsSerializerRule(StatboxSerializerRule):
    def __init__(self):
        super(StatboxPublicIdAliasOldPublicIdsSerializerRule, self).__init__(
            entity='old_public_id',
            operations=(ADDED, CHANGED, DELETED,),
            getter=None,
            formatter=None,
        )

    def make_entries(self, old_snapshot, new_snapshot, diff, action):
        old_logins = old_snapshot.public_id_alias.old_public_ids if old_snapshot else set()
        new_logins = new_snapshot.public_id_alias.old_public_ids if new_snapshot else set()

        added_logins = new_logins - old_logins
        removed_logins = old_logins - new_logins

        entries = []
        for operation, logins in (
            ('added', sorted(added_logins)),
            ('removed', sorted(removed_logins)),
        ):
            for login in logins:
                entries.append(
                    self._make_entry(
                        old_snapshot,
                        new_snapshot,
                        diff,
                        action,
                        operation,
                        type=ALIAS_NAME_TO_TYPE['old_public_id'],
                        old_public_id=login,
                    ),
                )
        return entries

    def is_applicable(self, old_snapshot, new_snapshot, diff):
        return 'public_id_alias' in diff.changed


class StatboxNewStyleAliasAddSerializerRule(StatboxSerializerRule):
    """Правило сериализации создания алиасов без связанных сидов"""
    def __init__(self, alias, field, formatter=default_formatter, model_attr=None):
        if model_attr is None:
            model_attr = self._field_to_model_attr(field)

        super(StatboxNewStyleAliasAddSerializerRule, self).__init__(
            entity=field,
            entity_alias='aliases',
            operations=(ADDED,),
            getter=None,
            formatter=formatter,
            model_attr=model_attr,
        )
        self.alias = alias

    def _field_to_model_attr(self, field):
        field = field.split('.')
        model_attr = field[:-1] + ['alias']
        return '.'.join(model_attr)

    def make_entries(self, old_snapshot, new_snapshot, diff, action):
        value = self._get_field_from_snapshot(self.model_attr, old_snapshot, new_snapshot)
        value = self.value_formatter(value)
        return [
            self._make_entry(
                old_snapshot,
                new_snapshot,
                diff,
                action,
                'added',
                type=ALIAS_NAME_TO_TYPE[self.alias],
                value=value,
            ),
        ]


class StatboxNewStyleAliasRmSerializerRule(StatboxSerializerRule):
    """Правило сериализации удаления алиасов без связанных сидов"""
    def __init__(self, alias, field):
        super(StatboxNewStyleAliasRmSerializerRule, self).__init__(
            entity=field,
            entity_alias='aliases',
            operations=(DELETED,),
            getter=None,
            formatter=None,
        )
        self.alias = alias

    def make_entries(self, old_snapshot, new_snapshot, diff, action):
        return [self._make_entry(
            old_snapshot,
            new_snapshot,
            diff,
            action,
            'removed',
            type=ALIAS_NAME_TO_TYPE[self.alias],
        )]


class StatboxAltDomainAliasSerializerRule(StatboxSerializerRule):
    def _base_entry(self, action, operation):
        return {
            'event': self.event,
            'operation': operation,
            'entity': self.aliases_entity,
        }

    def make_entries(self, old_snapshot, new_snapshot, diff, action, domain_id, operation):
        if domain_id in ALT_DOMAIN_ID_TO_SID:
            self.aliases_entity = 'subscriptions'
            return [self._make_entry(
                old_snapshot,
                new_snapshot,
                diff,
                action,
                operation,
                sid=ALT_DOMAIN_ID_TO_SID[domain_id],
            )]

        self.aliases_entity = 'aliases'
        return [self._make_entry(
            old_snapshot,
            new_snapshot,
            diff,
            action,
            operation,
            type=ALIAS_NAME_TO_TYPE['altdomain'],
            domain_id=domain_id,
        )]


class StatboxAltDomainAliasAddSerializerRule(StatboxAltDomainAliasSerializerRule):
    """Правило сериализации создания altdomain-алиасов"""
    def __init__(self):
        super(StatboxAltDomainAliasAddSerializerRule, self).__init__(
            entity='login',
            operations=(ADDED,),
            getter=None,
            formatter=None,
        )

    def make_entries(self, old_snapshot, new_snapshot, diff, action):
        domain_id = new_snapshot.altdomain_alias.domain_id
        return super(StatboxAltDomainAliasAddSerializerRule, self).make_entries(
            old_snapshot,
            new_snapshot,
            diff,
            action,
            domain_id,
            'added',
        )


class StatboxAltDomainAliasRmSerializerRule(StatboxAltDomainAliasSerializerRule):
    """Правило сериализации удаления altdomain-алиасов"""
    def __init__(self):
        super(StatboxAltDomainAliasRmSerializerRule, self).__init__(
            entity='',
            operations=(DELETED,),
            getter=None,
            formatter=None,
        )

    def make_entries(self, old_snapshot, new_snapshot, diff, action):
        if old_snapshot.altdomain_alias.alias is Undefined:
            return []

        domain_id = old_snapshot.altdomain_alias.domain_id
        return super(StatboxAltDomainAliasRmSerializerRule, self).make_entries(
            old_snapshot,
            new_snapshot,
            diff,
            action,
            domain_id,
            'removed',
        )


class StatboxUserDefinedLoginSerializerRule(StatboxSerializerRule):
    """Правило сериализации user_defined_login (пишется лишь если не совпадает с портальным алиасом)"""
    def __init__(self):
        super(StatboxUserDefinedLoginSerializerRule, self).__init__(
            entity='user_defined_login',
            operations=(ADDED,),
        )

    def make_entries(self, old_snapshot, new_snapshot, diff, action):
        if (
            new_snapshot.portal_alias and
            new_snapshot.user_defined_login and
            new_snapshot.user_defined_login != new_snapshot.normalized_login
        ):
            return super(StatboxUserDefinedLoginSerializerRule, self).make_entries(
                old_snapshot,
                new_snapshot,
                diff,
                action,
            )
        return []
