# -*- coding: utf-8 -*-

from passport.backend.core.serializers.logs.base import (
    ADDED,
    AttributeOfListItemSerializerRule,
    BaseSerializerRule,
    CHANGED,
    DELETED,
)
from passport.backend.core.serializers.logs.statbox.formatters import default_formatter
from passport.backend.core.serializers.logs.statbox.getters import identical_getter
from passport.backend.core.serializers.logs.utils import hierarchical_get
from six import iteritems


class StatboxSerializerRule(BaseSerializerRule):
    """Базовое правило сериализации изменений аккаунта для Statbox"""
    def __init__(self, entity, entity_alias=None, event='account_modification',
                 operations=(ADDED, CHANGED, DELETED),
                 getter=identical_getter, formatter=default_formatter,
                 snapshot_fields=None, extra_fields=None, model_attr=None):
        """
        entity: поле аккаунта, за изменением которого следим.
        entity_alias: имя поля для записи в лог. По умолчанию совпадает с entity.
        action: записывается в лог. Если не задано, используется events['action'].
        getter: функция для получения значения поля. Принимает объект, лежащий по пути, указанному в entity.
        formatter: функция для конвертации значения в строку.
        snapshot_fields: поля аккаунта, значения которых нужно дополнительно записать в лог.
        extra_fields: значения, которые нужно дополнительно записать в лог.
        """
        super(StatboxSerializerRule, self).__init__(
            entity=entity,
            entity_alias=entity_alias,
            operations=operations,
            model_attr=model_attr,
        )
        self.event = event
        self.snapshot_fields = snapshot_fields or {}
        self.extra_fields = extra_fields or {}
        self.value_getter = getter
        self.value_formatter = formatter

    def _get_operation_and_values(self, old_snapshot, new_snapshot):
        old = hierarchical_get(old_snapshot, self.model_attr)
        new = hierarchical_get(new_snapshot, self.model_attr)

        old = self.value_getter(old)
        new = self.value_getter(new)

        if old is None:
            operation = 'created'
        elif new is None:
            operation = 'deleted'
        else:
            operation = 'updated'

        return operation, old, new

    def _get_field_from_snapshot(self, field_path, old_snapshot, new_snapshot):
        old_field_value = hierarchical_get(old_snapshot, field_path)
        new_field_value = hierarchical_get(new_snapshot, field_path)
        return old_field_value or new_field_value or '-'

    def _get_fields_from_snapshot(self, old_snapshot, new_snapshot):
        result = {}
        for field_name, field_path in iteritems(self.snapshot_fields):
            result[field_name] = self._get_field_from_snapshot(field_path, old_snapshot, new_snapshot)
        return result

    def _base_entry(self, action, operation):
        return {
            'event': self.event,
            'operation': operation,
            'entity': self.entity_alias,
        }

    def _make_entry(self, old_snapshot, new_snapshot, diff, action, operation, **kwargs):
        entry = self._base_entry(action, operation)
        entry.update(self._get_fields_from_snapshot(old_snapshot, new_snapshot))
        entry.update(self.extra_fields)
        entry.update(**kwargs)
        return entry

    def make_entries(self, old_snapshot, new_snapshot, diff, action, **extra_fields):
        """Возвращает список лог-записей, созданных этим правилом"""
        operation, old, new = self._get_operation_and_values(old_snapshot, new_snapshot)
        return [self._make_entry(
            old_snapshot,
            new_snapshot,
            diff,
            action,
            operation,
            old=self.value_formatter(old),
            new=self.value_formatter(new),
            **extra_fields
        )]


class StatboxAttributeOfListItemSerializerRule(StatboxSerializerRule,
                                               AttributeOfListItemSerializerRule):
    def make_entries(self, old_snapshot, new_snapshot, diff, action,
                     entity_alias, entity_id, **extra_fields):
        operation, old, new = self._get_operation_and_values(old_snapshot, new_snapshot)
        return [self._make_entry(
            old_snapshot,
            new_snapshot,
            diff,
            action,
            operation,
            entity=entity_alias,
            entity_id=entity_id,
            old=self.value_formatter(old),
            new=self.value_formatter(new),
            **extra_fields
        )]

    def _base_entry(self, action, operation):
        return {
            'event': self.event,
            'operation': operation,
            'attribute': self.entity_alias,
        }
