# -*- coding: utf-8 -*-

from passport.backend.core.differ.differ import diff
from passport.backend.core.differ.types import Diff
from passport.backend.core.differ.utils import slice_diff
from passport.backend.core.serializers.logs.statbox.formatters import default_formatter
from passport.backend.core.serializers.logs.statbox.rules.base import StatboxSerializerRule
from passport.backend.core.serializers.logs.utils import hierarchical_get
from passport.backend.core.types.email.email import mask_email_for_statbox
from passport.backend.core.undefined import Undefined
from six import itervalues


class StatboxEmailsSerializerRule(StatboxSerializerRule):
    def __init__(self, entity, entity_alias=None, should_mask_email=True):
        self.should_mask_email = should_mask_email
        super(StatboxEmailsSerializerRule, self).__init__(entity, entity_alias=entity_alias)

    def make_entries(self, old_snapshot, new_snapshot, object_diff, action):
        emails_diff = slice_diff(object_diff, 'emails')
        changed_addresses = []

        if emails_diff.deleted is None and old_snapshot:
            emails_diff = Diff(
                emails_diff.added,
                emails_diff.changed,
                {},
            )

            changed_addresses = [
                email.address
                for email in itervalues(old_snapshot.emails._emails)
                if email.id is not Undefined
            ]

        emails_diff = slice_diff(emails_diff, '_emails')
        if not changed_addresses:
            changed_addresses = emails_diff.get_changed_fields()

        entries = []
        old_emails = hierarchical_get(old_snapshot, 'emails')
        new_emails = hierarchical_get(new_snapshot, 'emails')

        for address in changed_addresses:
            old = old_emails.find(address) if old_emails else None
            new = new_emails.find(address) if new_emails else None

            if old:
                email_id = old.id
            elif new:
                email_id = new.id

            if email_id is Undefined:
                continue

            if not new:
                instance = old
                fields_diff = diff(new, old)
            else:
                instance = new
                fields_diff = slice_diff(emails_diff, address)

            if self.should_mask_email:
                old_email = mask_email_for_statbox(old.address) if old else '-'
                new_email = mask_email_for_statbox(new.address) if new else '-'
            else:
                old_email = old.address if old else '-'
                new_email = new.address if new else '-'
            extra_fields = dict(
                email_id=email_id,
                old=old_email,
                new=new_email,
                is_suitable_for_restore=instance.is_suitable_for_restore,
            )

            if new is None:
                operation = 'deleted'
            elif old is None:
                operation = 'added'
            else:
                operation = 'updated'

            for field in fields_diff.get_changed_fields():
                value = getattr(instance, field)
                extra_fields[field] = default_formatter(value)

            for ignored_field in ('id', 'address'):
                extra_fields.pop(ignored_field, None)

            entry = self._make_entry(
                old_snapshot,
                new_snapshot,
                diff,
                action,
                operation,
                **extra_fields
            )
            entries.append(entry)

        return entries

    def is_applicable(self, old_snapshot, new_snapshot, diff):
        return 'emails' in diff.get_changed_fields()
