# -*- coding: utf-8 -*-

from functools import partial
from operator import attrgetter

from passport.backend.core.serializers.logs.base import AttributeOfListItemSerializerRule
from passport.backend.core.serializers.logs.statbox.formatters import format_phone_number
from passport.backend.core.serializers.logs.statbox.getters import build_getter
from passport.backend.core.serializers.logs.statbox.rules.base import StatboxSerializerRule
from passport.backend.core.serializers.logs.utils import hierarchical_get


class StatboxSecurePhoneSerializerRule(StatboxSerializerRule):
    def __init__(self, entity, should_mask_phone_number):
        _format_phone_number = partial(
            format_phone_number,
            should_mask=should_mask_phone_number,
        )
        super(StatboxSecurePhoneSerializerRule, self).__init__(
            entity=entity,
            entity_alias='phones.secure',
            model_attr='phones.secure',
            getter=build_getter(attrgetter('number')),
            formatter=_format_phone_number,
        )

    def _make_entry(self, old_snapshot, new_snapshot, diff, action, operation, **kwargs):
        old_id = hierarchical_get(old_snapshot, 'phones.secure_id') or '-'
        new_id = hierarchical_get(new_snapshot, 'phones.secure_id') or '-'
        return super(StatboxSecurePhoneSerializerRule, self)._make_entry(
            old_snapshot,
            new_snapshot,
            diff,
            action,
            operation,
            old_entity_id=old_id,
            new_entity_id=new_id,
            **kwargs
        )


class PharmaPhoneListSerializerRule(StatboxSerializerRule,
                                    AttributeOfListItemSerializerRule):
    def make_entries(self, old_snapshot, new_snapshot, diff, action,
                     entity_alias, entity_id, **extra_fields):
        new_number = hierarchical_get(new_snapshot, '{}.number'.format(self.entity))
        old_number = hierarchical_get(old_snapshot, '{}.number'.format(self.entity))

        entries = []
        new_confirmed = hierarchical_get(new_snapshot, '{}.confirmed'.format(self.entity))
        old_confirmed = hierarchical_get(old_snapshot, '{}.confirmed'.format(self.entity))
        if (new_confirmed != old_confirmed or new_number != old_number) and new_number is not None and new_confirmed is not None:
            entries.append({
                'action': 'confirmed',
                'phonenumber': new_number.digital,
            })

        return entries
