# -*- coding: utf-8 -*-

from functools import partial

from passport.backend.core.eav_type_mapping import ALIAS_NAME_TO_TYPE
from passport.backend.core.serializers.logs.base import (
    BaseListSerializer,
    BaseModelSerializer,
    CHANGED,
)
from passport.backend.core.serializers.logs.statbox.formatters import (
    disabled_status_formatter,
    format_phone_number,
    gender_formatter,
    mail_status_formatter,
    optional_datetime_formatter,
    shadowed_formatter,
)
from passport.backend.core.serializers.logs.statbox.getters import empty_as_none_getter
from passport.backend.core.serializers.logs.statbox.rules import (
    PharmaPhoneListSerializerRule,
    StatboxAliasAddSerializerRule,
    StatboxAliasRmSerializerRule,
    StatboxAltDomainAliasAddSerializerRule,
    StatboxAltDomainAliasRmSerializerRule,
    StatboxAttributeOfListItemSerializerRule,
    StatboxEmailsSerializerRule,
    StatboxFamilyKidDeleteRule,
    StatboxKarmaSerializerRule,
    StatboxNewStyleAliasAddSerializerRule,
    StatboxNewStyleAliasRmSerializerRule,
    StatboxPddAliasAdditionalLoginsSerializerRule,
    StatboxPublicIdAliasOldPublicIdsSerializerRule,
    StatboxSecurePhoneSerializerRule,
    StatboxSerializerRule,
    StatboxSubscriptionSerializerRule,
    StatboxUserDefinedLoginSerializerRule,
)
from passport.backend.core.serializers.logs.utils import hierarchical_get


class StatboxPlusSerializer(BaseModelSerializer):
    rules = (
        StatboxSerializerRule('enabled'),
        StatboxSerializerRule('trial_used_ts', formatter=optional_datetime_formatter),
        StatboxSerializerRule('subscription_stopped_ts', formatter=optional_datetime_formatter),
        StatboxSerializerRule('subscription_expire_ts', formatter=optional_datetime_formatter),
        StatboxSerializerRule('next_charge_ts', formatter=optional_datetime_formatter),
        StatboxSerializerRule('ott_subscription', getter=empty_as_none_getter),
        StatboxSerializerRule('family_role', getter=empty_as_none_getter),
        StatboxSerializerRule('cashback_enabled', getter=empty_as_none_getter),
        StatboxSerializerRule('subscription_level', getter=empty_as_none_getter),
        StatboxSerializerRule('is_frozen', getter=empty_as_none_getter),
        StatboxSerializerRule('subscriber_state', getter=empty_as_none_getter)
    )


class StatboxTakeoutSerializer(BaseModelSerializer):
    rules = (
        StatboxSerializerRule('extract_in_progress_since', formatter=optional_datetime_formatter),
    )


class StatboxPersonSerializer(BaseModelSerializer):
    # тут список туплов, чтобы не нарушать уже устоявшийся за много лет порядок логирования
    _rules_mapping = [
        ('firstname', StatboxSerializerRule('firstname')),
        ('lastname', StatboxSerializerRule('lastname')),
        ('language', StatboxSerializerRule('language')),
        ('country', StatboxSerializerRule('country')),
        ('gender', StatboxSerializerRule('gender', formatter=gender_formatter)),
        ('birthday', StatboxSerializerRule('birthday')),
        ('display_name', StatboxSerializerRule('display_name')),
        ('default_avatar', StatboxSerializerRule('default_avatar')),
        ('show_fio_in_public_name', StatboxSerializerRule('show_fio_in_public_name')),
        # ('fullname', ...) - fullname принимается в конструкторе, но явного сериализатора для него нет
    ]
    rules = None

    def __init__(self, entity=None, rules_to_include=None):
        # Возможность настройки логируемых полей при сериализации person.
        # В аргументах надо перечислить константы в виде массива, такие как firstname, lastname, ...
        # Если None, то логируется всё, что можно.
        rules_to_include = set() if rules_to_include is None else set(rules_to_include)

        self.rules = []

        # обхожу правила сериализации в объявленном выше порядке, чтобы сохранить
        # устоявшийся за время порядок логирования изменения полей на аккаунте
        for rule_name, serializer_rule in self._rules_mapping:
            if rule_name in rules_to_include or not rules_to_include:
                self.rules.append(serializer_rule)

        if 'fullname' in rules_to_include or not rules_to_include:
            self._log_fullname = True
        else:
            self._log_fullname = False

        super(StatboxPersonSerializer, self).__init__(entity=entity)

    def _get_full_name_from_snapshot(self, snapshot):
        firstname = hierarchical_get(snapshot, 'person.firstname')
        lastname = hierarchical_get(snapshot, 'person.lastname')
        if firstname and lastname:
            return u'{} {}'.format(firstname, lastname)
        elif firstname:
            return firstname
        elif lastname:
            return lastname

    def serialize(self, old_snapshot, new_snapshot, diff, entity_prefix=None, **kwargs):
        result = super(StatboxPersonSerializer, self).serialize(
            old_snapshot,
            new_snapshot,
            diff,
            entity_prefix=entity_prefix,
            **kwargs
        )

        if not self._log_fullname:
            # дальнейшие вычисления для логирования fullname не нужны => выход из функции
            return result

        firstname_rule = [r for r in self.rules if r.entity == 'firstname'][0]
        lastname_rule = [r for r in self.rules if r.entity == 'lastname'][0]
        if firstname_rule.clone_for_prefix('person').is_applicable(
            old_snapshot, new_snapshot, diff
        ) or lastname_rule.clone_for_prefix('person').is_applicable(old_snapshot, new_snapshot, diff):
            old_fullname = self._get_full_name_from_snapshot(old_snapshot)
            new_fullname = self._get_full_name_from_snapshot(new_snapshot)
            if old_fullname is None:
                operation = 'created'
            elif new_fullname is None:
                operation = 'deleted'
            else:
                operation = 'updated'
            result.append(
                dict(
                    entity='person.fullname',
                    event='account_modification',
                    operation=operation,
                    old=old_fullname or '-',
                    new=new_fullname or '-',
                )
            )
        return result


class StatboxPasswordSerializer(BaseModelSerializer):
    rules = (
        StatboxSerializerRule(
            'encrypted_password',
            entity_alias='password.encrypted',
            formatter=shadowed_formatter,
        ),
        StatboxSerializerRule('encoding_version'),
        StatboxSerializerRule('quality'),
        StatboxSerializerRule('forced_changing_reason', entity_alias='password.is_changing_required'),
        StatboxSerializerRule('pwn_forced_changing_suspended_at'),
    )


class StatboxPddDomainSerializer(BaseModelSerializer):
    rules = (
        StatboxSerializerRule(
            'domain',
            entity_alias='domain_name',
        ),
        StatboxSerializerRule(
            'id',
            entity_alias='domain_id',
        ),
        StatboxSerializerRule(
            'organization_name',
            entity_alias='domain_organization_name',
        ),
        StatboxSerializerRule(
            'display_master_id',
            entity_alias='domain_display_master_id',
        ),
    )

    def serialize(self, old_snapshot, new_snapshot, diff, entity_prefix=None, **kwargs):
        result = super(StatboxPddDomainSerializer, self).serialize(
            old_snapshot,
            new_snapshot,
            diff,
            entity_prefix=entity_prefix,
            **kwargs
        )

        for entry in result:
            if entry['entity'] == 'domain_name':
                entry['new'] = entry['new'].encode('idna')
                entry['old'] = entry['old'].encode('idna')

        return result


class StatboxHintSerializer(BaseModelSerializer):
    rules = (
        StatboxSerializerRule('question', formatter=shadowed_formatter),
        StatboxSerializerRule('answer', formatter=shadowed_formatter),
    )


class StatboxKarmaSerializer(BaseModelSerializer):
    rules = (
        StatboxKarmaSerializerRule(),
    )


class StatboxSubscriptionsSerializer(BaseListSerializer):
    entity = 'subscriptions'
    list_item_entity_alias = 'subscriptions'

    rules = (StatboxSubscriptionSerializerRule(),)


class StatboxAltDomainAliasSerializer(BaseModelSerializer):
    rules = (
        StatboxAltDomainAliasAddSerializerRule(),
        StatboxAltDomainAliasRmSerializerRule(),
    )


class StatboxPhonenumberAliasSerializer(BaseModelSerializer):
    rules = (
        StatboxAliasAddSerializerRule('number', sid=65),
        StatboxAliasRmSerializerRule(sid=65),
        StatboxSerializerRule(
            'number',
            entity_alias='aliases',
            operations=(CHANGED,),
            extra_fields={'type': str(ALIAS_NAME_TO_TYPE['phonenumber'])},
            formatter=shadowed_formatter,
        ),
        StatboxSerializerRule('enable_search'),
    )


class StatboxYandexoidAliasSerializer(BaseModelSerializer):
    rules = (
        StatboxAliasAddSerializerRule('login', sid=669),
        StatboxAliasRmSerializerRule(sid=669),
    )


class StatboxPhoneListSerializer(BaseListSerializer):
    entity = '_phones'
    list_item_entity_alias = 'phone'

    def __init__(self, should_mask_phone_number):
        self.should_mask_phone_number = should_mask_phone_number

    @property
    def rules(self):
        return (
            StatboxAttributeOfListItemSerializerRule(
                'number',
                entity_alias='number',
                formatter=partial(
                    format_phone_number,
                    should_mask=self.should_mask_phone_number,
                ),
            ),
            StatboxAttributeOfListItemSerializerRule('confirmed', entity_alias='confirmed'),
            StatboxAttributeOfListItemSerializerRule('_bound', entity_alias='bound'),
            StatboxAttributeOfListItemSerializerRule('secured', entity_alias='secured'),
            StatboxAttributeOfListItemSerializerRule('admitted', entity_alias='admitted'),
        )


class CryptastatPhonesSerializer(BaseModelSerializer):
    rules = [
        StatboxSecurePhoneSerializerRule(
            'secure_id',
            should_mask_phone_number=False,
        ),
    ]

    child_serializers = [
        StatboxPhoneListSerializer(should_mask_phone_number=False),
    ]

    def __init__(self):
        super(CryptastatPhonesSerializer, self).__init__('phones')


class PharmaPhoneListSerializer(StatboxPhoneListSerializer):
    @property
    def rules(self):
        return (PharmaPhoneListSerializerRule(''),)


class PharmaPhonesSerializer(BaseModelSerializer):
    child_serializers = [
        PharmaPhoneListSerializer(should_mask_phone_number=False),
    ]

    def __init__(self):
        super(PharmaPhonesSerializer, self).__init__('phones')


class StatboxPhoneOperationsSerializer(object):
    def serialize(self, old_snapshot, new_snapshot, diff, entity_prefix=None, **kwargs):
        entries = []

        if old_snapshot and old_snapshot.phones:
            old_operations = {op.id: op for op in old_snapshot.phones.get_logical_operations(None)}
        else:
            old_operations = {}

        if new_snapshot and new_snapshot.phones:
            new_operations = {op.id: op for op in new_snapshot.phones.get_logical_operations(None)}
        else:
            new_operations = {}

        for _id in new_operations:
            if _id not in old_operations:
                entries += new_operations[_id].statbox_entries_for_create()

        return entries


class StatboxAccountInfosecSerializer(BaseModelSerializer):
    rules = (
        StatboxSerializerRule(
            'global_logout_datetime',
            entity_alias='account.global_logout_datetime',
        ),
        StatboxSerializerRule(
            'tokens_revoked_at',
            entity_alias='account.revoker.tokens',
        ),
        StatboxSerializerRule(
            'web_sessions_revoked_at',
            entity_alias='account.revoker.web_sessions',
        ),
        StatboxSerializerRule(
            'app_passwords_revoked_at',
            entity_alias='account.revoker.app_passwords',
        ),
        StatboxSerializerRule(
            'disabled_status',
            entity_alias='account.disabled_status',
            formatter=disabled_status_formatter,
        ),
        StatboxSerializerRule(
            'mail_status',
            entity_alias='account.mail_status',
            formatter=mail_status_formatter,
        ),
        StatboxUserDefinedLoginSerializerRule(),
        StatboxNewStyleAliasAddSerializerRule('portal', 'portal_alias.login'),
        StatboxNewStyleAliasAddSerializerRule('pdd', 'pdd_alias.email'),
        StatboxNewStyleAliasAddSerializerRule('lite', 'lite_alias.email'),
        StatboxNewStyleAliasAddSerializerRule('scholar', 'scholar_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('social', 'social_alias.login'),
        StatboxNewStyleAliasAddSerializerRule('phonish', 'phonish_alias.login'),
        StatboxNewStyleAliasAddSerializerRule('neophonish', 'neophonish_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('mailish', 'mailish_alias.mailish_id'),
        StatboxNewStyleAliasAddSerializerRule('kinopoisk', 'kinopoisk_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('uber', 'uber_alias.uber_id'),
        StatboxNewStyleAliasAddSerializerRule('yambot', 'yambot_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('kolonkish', 'kolonkish_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('public_id', 'public_id_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('kiddish', 'kiddish_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('bank_phonenumber', 'bank_phonenumber_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('federal', 'federal_alias.alias'),
        StatboxSerializerRule(
            'mailish_alias.mailish_id',
            entity_alias='aliases',
            operations=(CHANGED,),
            extra_fields={'type': str(ALIAS_NAME_TO_TYPE['mailish'])},
        ),
        StatboxSerializerRule(
            'public_id_alias.alias',
            entity_alias='aliases',
            operations=(CHANGED,),
            extra_fields={'type': str(ALIAS_NAME_TO_TYPE['public_id'])},
        ),
        StatboxNewStyleAliasRmSerializerRule('portal', 'portal_alias'),
        StatboxNewStyleAliasRmSerializerRule('pdd', 'pdd_alias'),
        StatboxNewStyleAliasRmSerializerRule('lite', 'lite_alias'),
        StatboxNewStyleAliasRmSerializerRule('scholar', 'scholar_alias'),
        StatboxNewStyleAliasRmSerializerRule('social', 'social_alias'),
        StatboxNewStyleAliasRmSerializerRule('phonish', 'phonish_alias'),
        StatboxNewStyleAliasRmSerializerRule('neophonish', 'neophonish_alias'),
        StatboxNewStyleAliasRmSerializerRule('mailish', 'mailish_alias'),
        StatboxNewStyleAliasRmSerializerRule('kinopoisk', 'kinopoisk_alias'),
        StatboxNewStyleAliasRmSerializerRule('uber', 'uber_alias'),
        StatboxNewStyleAliasRmSerializerRule('yambot', 'yambot_alias'),
        StatboxNewStyleAliasRmSerializerRule('kolonkish', 'kolonkish_alias'),
        StatboxNewStyleAliasRmSerializerRule('public_id', 'public_id_alias'),
        StatboxNewStyleAliasRmSerializerRule('kiddish', 'kiddish_alias'),
        StatboxNewStyleAliasRmSerializerRule('bank_phonenumber', 'bank_phonenumber_alias'),
        StatboxNewStyleAliasRmSerializerRule('federal', 'federal_alias'),
        StatboxNewStyleAliasAddSerializerRule(
            'phonenumber',
            'phonenumber_alias.number',
        ),
        StatboxNewStyleAliasRmSerializerRule(
            'phonenumber',
            'phonenumber_alias',
        ),
        StatboxEmailsSerializerRule('emails', 'account.emails', should_mask_email=False),
        StatboxSecurePhoneSerializerRule(
            'phones.secure_id',
            should_mask_phone_number=False,
        ),
    )

    child_serializers = (
        StatboxPersonSerializer(
            'person',
            rules_to_include=[
                'firstname',
                'lastname',
                'birthday',
                'language',
                'country',
                'gender',
                'fullname',
            ],
        ),
        StatboxSubscriptionsSerializer(),
        StatboxPhoneOperationsSerializer(),
        StatboxAltDomainAliasSerializer('altdomain_alias'),
        StatboxPhonenumberAliasSerializer('phonenumber_alias'),
        StatboxYandexoidAliasSerializer('yandexoid_alias'),
    )

    def serialize(self, old_snapshot, new_snapshot, diff, entity_prefix=None, **kwargs):
        return super(StatboxAccountInfosecSerializer, self).serialize(
            old_snapshot, new_snapshot, diff, entity_prefix=entity_prefix, **kwargs
        )


class StatboxAccountSerializer(BaseModelSerializer):
    rules = (
        StatboxSerializerRule(
            'enable_app_password',
            entity_alias='account.enable_app_password',
        ),
        StatboxSerializerRule(
            'global_logout_datetime',
            entity_alias='account.global_logout_datetime',
        ),
        StatboxSerializerRule(
            'tokens_revoked_at',
            entity_alias='account.revoker.tokens',
        ),
        StatboxSerializerRule(
            'web_sessions_revoked_at',
            entity_alias='account.revoker.web_sessions',
        ),
        StatboxSerializerRule(
            'app_passwords_revoked_at',
            entity_alias='account.revoker.app_passwords',
        ),
        StatboxSerializerRule(
            'disabled_status',
            entity_alias='account.disabled_status',
            formatter=disabled_status_formatter,
        ),
        StatboxSerializerRule(
            'mail_status',
            entity_alias='account.mail_status',
            formatter=mail_status_formatter,
        ),
        StatboxUserDefinedLoginSerializerRule(),
        StatboxNewStyleAliasAddSerializerRule('portal', 'portal_alias.login'),
        StatboxNewStyleAliasAddSerializerRule('pdd', 'pdd_alias.email'),
        StatboxNewStyleAliasAddSerializerRule('lite', 'lite_alias.email'),
        StatboxNewStyleAliasAddSerializerRule('scholar', 'scholar_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('social', 'social_alias.login'),
        StatboxNewStyleAliasAddSerializerRule('phonish', 'phonish_alias.login'),
        StatboxNewStyleAliasAddSerializerRule('neophonish', 'neophonish_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('mailish', 'mailish_alias.mailish_id'),
        StatboxNewStyleAliasAddSerializerRule('kinopoisk', 'kinopoisk_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('uber', 'uber_alias.uber_id'),
        StatboxNewStyleAliasAddSerializerRule('yambot', 'yambot_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('kolonkish', 'kolonkish_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('public_id', 'public_id_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('kiddish', 'kiddish_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('bank_phonenumber', 'bank_phonenumber_alias.alias'),
        StatboxNewStyleAliasAddSerializerRule('federal', 'federal_alias.alias'),
        StatboxSerializerRule(
            'mailish_alias.mailish_id',
            entity_alias='aliases',
            operations=(CHANGED,),
            extra_fields={'type': str(ALIAS_NAME_TO_TYPE['mailish'])},
        ),
        StatboxSerializerRule(
            'public_id_alias.alias',
            entity_alias='aliases',
            operations=(CHANGED,),
            extra_fields={'type': str(ALIAS_NAME_TO_TYPE['public_id'])},
        ),
        StatboxNewStyleAliasRmSerializerRule('portal', 'portal_alias'),
        StatboxNewStyleAliasRmSerializerRule('pdd', 'pdd_alias'),
        StatboxNewStyleAliasRmSerializerRule('lite', 'lite_alias'),
        StatboxNewStyleAliasRmSerializerRule('scholar', 'scholar_alias'),
        StatboxNewStyleAliasRmSerializerRule('social', 'social_alias'),
        StatboxNewStyleAliasRmSerializerRule('phonish', 'phonish_alias'),
        StatboxNewStyleAliasRmSerializerRule('neophonish', 'neophonish_alias'),
        StatboxNewStyleAliasRmSerializerRule('mailish', 'mailish_alias'),
        StatboxNewStyleAliasRmSerializerRule('kinopoisk', 'kinopoisk_alias'),
        StatboxNewStyleAliasRmSerializerRule('uber', 'uber_alias'),
        StatboxNewStyleAliasRmSerializerRule('yambot', 'yambot_alias'),
        StatboxNewStyleAliasRmSerializerRule('kolonkish', 'kolonkish_alias'),
        StatboxNewStyleAliasRmSerializerRule('public_id', 'public_id_alias'),
        StatboxNewStyleAliasRmSerializerRule('kiddish', 'kiddish_alias'),
        StatboxNewStyleAliasRmSerializerRule('bank_phonenumber', 'bank_phonenumber_alias'),
        StatboxNewStyleAliasRmSerializerRule('federal', 'federal_alias'),
        StatboxNewStyleAliasAddSerializerRule(
            'phonenumber',
            'phonenumber_alias.number',
            shadowed_formatter,
        ),
        StatboxNewStyleAliasRmSerializerRule(
            'phonenumber',
            'phonenumber_alias',
        ),
        StatboxEmailsSerializerRule('emails', 'account.emails'),
        StatboxPddAliasAdditionalLoginsSerializerRule(),
        StatboxPublicIdAliasOldPublicIdsSerializerRule(),
        StatboxSecurePhoneSerializerRule(
            'phones.secure_id',
            should_mask_phone_number=True,
        ),
        StatboxSerializerRule(
            'content_rating_class',
            entity_alias='account.content_rating_class',
        ),
        StatboxSerializerRule(
            'magic_link_login_forbidden',
            entity_alias='account.magic_link_login_forbidden',
        ),
        StatboxSerializerRule(
            'music_content_rating_class',
            entity_alias='account.music_content_rating_class',
        ),
        StatboxSerializerRule(
            'qr_code_login_forbidden',
            entity_alias='account.qr_code_login_forbidden',
        ),
        StatboxSerializerRule(
            'sms_code_login_forbidden',
            entity_alias='account.sms_code_login_forbidden',
        ),
        StatboxSerializerRule(
            'takeout.subscription',
            entity_alias='takeout.subscription',
        ),
        StatboxSerializerRule(
            'video_content_rating_class',
            entity_alias='account.video_content_rating_class',
        ),
        StatboxSerializerRule(
            'billing_features',
            entity_alias='account.billing_features',
        ),
        StatboxSerializerRule(
            'force_challenge',
            entity_alias='account.force_challenge',
        ),
        StatboxSerializerRule(
            'user_defined_public_id',
            entity_alias='account.user_defined_public_id',
        ),
        StatboxSerializerRule(
            'sms_2fa_on',
            entity_alias='account.sms_2fa_on',
        ),
        StatboxSerializerRule(
            'takeout.delete.subscription',
            entity_alias='takeout.delete.subscription',
        ),
        StatboxFamilyKidDeleteRule(),
        StatboxSerializerRule(
            'unsubscribed_from_maillists',
            entity_alias='account.unsubscribed_from_maillists',
        ),
        StatboxSerializerRule(
            'personal_data_public_access_allowed',
            entity_alias='account.personal_data_public_access_allowed',
        ),
        StatboxSerializerRule(
            'personal_data_third_party_processing_allowed',
            entity_alias='account.personal_data_third_party_processing_allowed',
        ),
        StatboxSerializerRule(
            entity='scholar_password.serialized_encrypted_password',
            entity_alias='account.scholar_password',
            formatter=shadowed_formatter,
        ),
        StatboxSerializerRule(
            entity='family_pay_enabled',
            entity_alias='account.family_pay.enabled',
        ),
        StatboxSerializerRule(
            entity='is_documents_agreement_accepted',
            entity_alias='account.is_documents_agreement_accepted',
        ),
        StatboxSerializerRule(
            entity='is_dzen_sso_prohibited',
            entity_alias='account.is_dzen_sso_prohibited',
        ),
        StatboxSerializerRule(
            entity='last_child_family',
            entity_alias='account.last_child_family',
        ),
        StatboxSerializerRule(
            entity='can_manage_children',
            entity_alias='account.can_manage_children',
        ),
    )

    child_serializers = (
        StatboxPersonSerializer('person'),
        StatboxPddDomainSerializer('domain'),
        StatboxPasswordSerializer('password'),
        StatboxHintSerializer('hint'),
        StatboxKarmaSerializer('karma'),
        StatboxSubscriptionsSerializer(),
        StatboxPhoneOperationsSerializer(),
        StatboxPlusSerializer('plus'),
        StatboxTakeoutSerializer('takeout'),
        StatboxAltDomainAliasSerializer('altdomain_alias'),
        StatboxPhonenumberAliasSerializer('phonenumber_alias'),
        StatboxYandexoidAliasSerializer('yandexoid_alias'),
    )


class StatboxFamilyMembersSerializer(BaseListSerializer):
    entity = 'members'
    list_item_entity_alias = 'members'

    rules = (
        StatboxAttributeOfListItemSerializerRule('uid'),
    )


class StatboxFamilyKidsSerializer(BaseListSerializer):
    entity = 'kids.members'
    list_item_entity_alias = 'kid'

    rules = (
        StatboxAttributeOfListItemSerializerRule('uid', entity_alias='uid'),
    )


class StatboxFamilySerializer(BaseModelSerializer):
    rules = (
        StatboxSerializerRule('admin_uid'),
    )

    child_serializers = (
        StatboxFamilyMembersSerializer(),
        StatboxFamilyKidsSerializer(),
    )
