# -*- coding: utf-8 -*-

from datetime import (
    datetime,
    timedelta,
)
import json

from passport.backend.core.db.schemas import tracks_table
from passport.backend.core.db.utils import delete_with_limit
from passport.backend.core.models.persistent_track import PersistentTrack
from passport.backend.core.serializers.base import DirectSerializer
from passport.backend.core.serializers.query import (
    GenericQuery,
    GenericShardedDeleteQuery,
    GenericShardedInsertQuery,
    GenericShardedUpdateQuery,
)


class PersistentTrackSerializer(DirectSerializer):
    model = PersistentTrack
    table = tracks_table

    transactional = False

    def extract_data(self, old, new, fields):
        data = super(PersistentTrackSerializer, self).extract_data(old, new, fields)

        if 'content' in data:
            data['content'] = json.dumps(data['content'], sort_keys=True)
        return data

    def create(self, old, new, data):
        return [
            GenericShardedInsertQuery(self.table, new.uid, data),
        ]

    def change(self, old, new, data):
        track_id = old.track_id
        return [
            GenericShardedUpdateQuery(
                self.table,
                new.uid,
                None,
                data,
                filter_by=(self.table.c.track_id == track_id.encode('utf8')),
            ),
        ]

    def delete(self, old):
        track_id = old.track_id
        return [
            GenericShardedDeleteQuery(
                self.table,
                old.uid,
                None,
                filter_by=(self.table.c.track_id == track_id.encode('utf8')),
            ),
        ]


class PersistentTrackCleanupQuery(GenericQuery):
    def __init__(self, limit, offset=0):
        super(PersistentTrackCleanupQuery, self).__init__(
            table=tracks_table,
            data=None,
        )
        self.limit = limit
        self.offset = offset

    def to_query(self):
        expired_before = datetime.now() - timedelta(seconds=self.offset)
        return delete_with_limit(self.table, self.limit).where(self.table.c.expired <= expired_before)
