# -*- coding: utf-8 -*-

from passport.backend.core.db.query import DbQuery
from passport.backend.core.db.utils import encode_params_for_db


class GenericQuery(DbQuery):
    def __init__(self, table, data):
        self.table = table
        self.data = data

    def get_table(self):
        return self.table

    def get_pk_field(self):
        if self.table is None:
            raise RuntimeError("No table specified!")

        pks = list(self.table.primary_key)
        if not pks:
            raise RuntimeError('No PK found on table "%s"' % self.table.name)
        return pks[0]


class GenericInsertQuery(GenericQuery):
    def to_query(self):
        return self.table.insert().values(**encode_params_for_db(self.data))


class GenericUpdateQuery(GenericQuery):
    def __init__(self, table, model_id, data, filter_by=None):
        super(GenericUpdateQuery, self).__init__(table, data)
        self.model_id = model_id
        self.filter_by = filter_by

    def to_query(self):
        q = self.table.update().values(
            **encode_params_for_db(self.data)
        )
        if self.filter_by is None:
            q = q.where(self.get_pk_field() == self.model_id)
        else:
            q = q.where(self.filter_by)
        return q


class GenericDeleteQuery(GenericQuery):
    def __init__(self, table, model_id, filter_by=None):
        super(GenericDeleteQuery, self).__init__(table, None)
        self.model_id = model_id
        self.filter_by = filter_by

    def to_query(self):
        # TODO: сделать поддержку композитных ключей
        q = self.table.delete()
        if self.filter_by is None:
            q = q.where(self.get_pk_field() == self.model_id)
        else:
            q = q.where(self.filter_by)
        return q


class GenericShardedInsertQuery(GenericInsertQuery):
    _SHARDED = True

    def __init__(self, table, uid, data):
        self.uid = uid
        super(GenericShardedInsertQuery, self).__init__(table, data)


class GenericShardedUpdateQuery(GenericUpdateQuery):
    _SHARDED = True

    def __init__(self, table, uid, model_id, data, filter_by=None):
        self.uid = uid
        super(GenericShardedUpdateQuery, self).__init__(table, model_id, data, filter_by=filter_by)


class GenericShardedDeleteQuery(GenericDeleteQuery):
    _SHARDED = True

    def __init__(self, table, uid, model_id, filter_by=None):
        self.uid = uid
        super(GenericShardedDeleteQuery, self).__init__(table, model_id, filter_by=filter_by)
