# -*- coding: utf-8 -*-
from passport.backend.core.db.query import DbTransaction
from passport.backend.core.db.schemas import pdd_domains_table
from passport.backend.core.serializers.base import Serializer
from passport.backend.core.serializers.domain import (
    build_insert_domain_event_query,
    prepare_domain_name,
)
from passport.backend.core.serializers.query import GenericUpdateQuery


class SwapDomainsBatchSerializer(Serializer):
    def serialize(self, old, new, difference):
        if old or not new:
            raise ValueError('SwapDomains batches can not be updated or deleted')

        return self.create(batch=new)

    def create(self, batch):
        if not batch.master_domain or not batch.alias:
            raise ValueError('You must specify both domains')
        if batch.alias.domain not in batch.master_domain.aliases:
            raise ValueError('Domain can be swapped only with its alias')

        return [batch_to_queries(batch)]


@DbTransaction
def batch_to_queries(batch):
    table = pdd_domains_table
    queries = []

    # Нормализуем имена доменов
    old_master_name = prepare_domain_name(batch.master_domain.domain).encode('utf8')
    old_alias_name = prepare_domain_name(batch.alias.domain).encode('utf8')

    # Просто свопнуть значения нельзя из-за ограничения уникальности домена,
    # а MySQL не умеет deferred constraints.
    # Поэтому приходится идти через фейковое промежуточное значение.
    queries += [
        GenericUpdateQuery(table, batch.alias.id, {'name': b'##%s##' % old_alias_name}),
        GenericUpdateQuery(table, batch.master_domain.id, {'name': old_alias_name}),
        GenericUpdateQuery(table, batch.alias.id, {'name': old_master_name}),
    ]
    # Запишем в таблицу событий, чтобы ЧЯ перечитал из БД нужные данные
    queries += [
        build_insert_domain_event_query('swap', batch.master_domain.id, meta=batch.alias.id),
        build_insert_domain_event_query('update', batch.alias.id),
    ]
    return queries
