# -*- coding: utf-8 -*-

"""
Человеческое описание сервиса есть на странице http://wiki.yandex-team.ru/passport/sids.
У сервиса есть айдишник (sid).
Отдельная сущность подписка. Аккаунт имеет подписку на сервис.
Задавание доменов (url) нужно продумать! Связано с локализацией.

Есть читабельные имена на разных языках.
Обычно прокидывается из танкера на уровне шаблонов.
И возможно здесь это будет ненужно.

Политики подписывания и отписки.
Есть такое свойство, которое запрещает/разрешает пользователю подписываться на сервис.

Является ли сервис удаляемым или блокирующим.
Сервисы вроде как блокирующие, при удалении аккаунт.
Аккаунт вместо удаления блокируется.
Хотим избавиться и сделать удаление аккаунта, но неудаляемость подписки.

Доступность сервиса для пддшника.

Подтверждение соглашения, для подписки на сервис. Посмотреть как реализовано.
"""


from six import iteritems


class Service(object):
    """
    У сервиса может быть несколько from-ов, и может быть главный from
    """
    # Надо обновлять функцию from_service, если у сервиса появляются новые свойства
    def __init__(self, sid=None, description=None, slug=None, url=None, for_pdd=False,
                 stores_not_uid_in_login=False,
                 requires_login=False, requires_password=False, internal_subscribe=False,
                 is_renewable=False, is_real=False):
        self.sid = sid
        self.description = description
        self.slug = slug
        self.url = url
        self.for_pdd = for_pdd
        self.stores_not_uid_in_login = stores_not_uid_in_login
        self.requires_login = requires_login
        self.requires_password = requires_password
        self.internal_subscribe = internal_subscribe
        self.is_renewable = is_renewable
        self.is_real = is_real

    def __eq__(self, obj):
        if isinstance(obj, Service):
            return self.sid == obj.sid
        return False

    def __hash__(self):
        return self.sid if isinstance(self.sid, int) else 0

    def __str__(self):
        return str(self.sid)

    @staticmethod
    def by_sid(sid):
        return get_service(sid=sid)

    @staticmethod
    def by_slug(slug):
        return get_service(slug=slug)

    @staticmethod
    def from_service(service):
        return Service(
            sid=service.sid,
            description=service.description,
            slug=service.slug,
            url=service.url,
            for_pdd=service.for_pdd,
            stores_not_uid_in_login=service.stores_not_uid_in_login,
            requires_login=service.requires_login,
            requires_password=service.requires_password,
            internal_subscribe=service.internal_subscribe,
            is_renewable=service.is_renewable,
            is_real=service.is_real,
        )

    def __repr__(self):
        return "<%s: %r>" % (self.__class__.__name__, self.__dict__)


# "Настоящие" сервисы
REAL_SERVICES = {
    'mail': 2,
    'fotki': 5,
    'cards': 6,
    'podpiska': 9,
    'dz': 12,
    'slova': 14,  # Директ
    'narodmail': 16,
    'sfera': 18,  # не используется
    'balance': 19,  # баланс-билинг
    'lenta': 23,
    'partner': 24,  # рекламная сеть
    'market': 25,
    'jabber': 27,  # джаббер (online)
    'tv': 29,
    'afisha': 30,
    'calendar': 31,
    'kuda': 38,
    'video': 39,
    'kraski': 40,
    'slovari': 41,
    'disk': 44,
    'pvo': 45,  # сервис закрыт, сид удалён
    'webmaster': 47,
    'metrika': 48,
    'maps': 49,
    'musicpartner': 50,  # музыка для партнёров
    'wikimaps': 52,  # народные карты
    'xml': 53,  # xml.yandex.ru
    'music': 54,
    'site_search': 56,  # поиск по сайту
    'cloud': 59,  # чемодан (диск?)
    'newspartner': 60,  # новости, партнёрский интерфейс. Неудаляемый
    'galatasaray': 61,
    'rabota': 72,  # Работа
    'realty': 73,  # Недвижимость
    'gazeta': 76,  # газета, социальные новости
    'mobilecloud': 77,  # мобильный я.диск
    'mobilemusic': 78,  # мобильная музыка
    'contest': 79,
    'cloudsoft': 80,  # будет проставлять я.диск, если человек установил их ПО
    'mynews': 83,  # Мои Новости
    'storedevconsole': 84,  # Кабинет разработчика приложений Яндекс.1Store
    'pi': 85,  # Партнерский интерфейс
    'publicdemo': 86,  # Для временных кампаний, промостраницы и тп
    'gamification': 87,
    'livemail': 88,  # Живые письма
    'postoffice': 91,  # Почтовый офис
    'cloudmigrant': 93,  # Дополнительный сид для диска
    'videosearch': 95,  # Видео-поиск
    'avia': 97,  # Авиабилеты
    'kinopoisk': 98,  # Кинопоиск
    'realty-agent': 106,  # Недвижимость -- агент по недвижимости
    'rabota-employee': 108,  # Все соискатели сервиса Работа
    'rabota-blue-collar': 109,  # "синие воротнички" сервиса Работа
    'rabota-employer': 110,  # Работодатели сервиса Работа
    'rabota-mobile-app': 111,  # Пользователи мобильного приложения сервиса Работа
    'rabota-mobile-site': 112,  # Пользователи мобильного сайта сервиса Работа
    'realty-moderator': 113,  # Модератор Недвижимости
    'taxi': 114,  # Яндекс.Такси
    'toloka': 116,  # Яндекс.Толока
    'telephony': 117,  # Яндекс.Телефония
    'drive': 118,  # Драйв
    'investments': 119,  # Инвестиции
    'telemost': 121,  # Телемост
    'mailpro': 122,  # Почта Про
    'mailpro_on': 123,  # Почта Про принудительное включение
    'mailpro_off': 124,  # Почта Про принудительное выключение (перекрывает mailpro_on)
    'bank': 125,  # Я.Банк
}


_services = {}
_services.update(REAL_SERVICES)

# Всякие служебные сервисы
_services.update(
    {
        'dev': '',
        'passport': 8,
        'so': 17,  # спамоборона
        'mpop': 22,  # не используется
        'friends': 26,  # человек имеет >=2 ку или создал аккаунт через ярушку
        'light': 33,
        'yasms': 36,  # для выставления провалидированного телефона пользователю
        'public': 37,  # у пользователя есть фотки или дневник
        'wwwdgt': 42,
        'radiojam': 51,  # радиосводки о пробках (???)
        'mobilmail': 55,  # пользователь засветился с мобильным приложением яндекс.почты
        'wdgt': 57,
        'social': 58,  # социальный пользователь
        'cloudpaid': 64,  # оплачен диск
        'phonenumber-alias': 65,
        'strongpwd': 67,
        'phonish': 68,  # добавлен для проекта Парковки
        'tune': 70,  # нужен только mode=admchangereg и записи в historydb, подписка пользователей не предполагается
        'yandsearch': 74,  # Большой Поиск, сид нужен для admchangereg'а, подписка пользователей не предполагается
        'cloudweb': 81,  # диск показывает вкладку в вебе для пользователей, которые пришли в диск до выхода приложений
        'mobileproxy': 82,  # PASSP-3687, будет использоваться "мобильной проксёй"
        'islandsapi': 90,  # API островов
        'bugbounty': 96,  # Охота
        'changepasswordonnextlogon': 100,
        'pddeula': 102,
        'pddadmin': 104,
        'mailapp2014-androidphone': 201,  # Новая мобильная почта, телефон на Android
        'mailapp2014-androidtablet': 202,  # Новая мобильная почта, планшет на Android
        'mailapp2014-iphone': 203,  # Новая мобильная почта, iPhone
        'mailapp2014-ipad': 204,  # Новая мобильная почта, iPad
        'mailapp2014-ipadbirdseye': 205,  # Новая мобильная почта, iPad-Birdseye
        'realty-app-ios': 206,  # Яндекс.Недвижимость, приложение на iOS
        'realty-app-android': 207,  # Яндекс.Недвижимость, приложение на Android
        'mailexp': 666,  # подопытный почтовый кролик
        'mailtest': 667,  # подопытный почтовый кролик
        'betatest': 668,
        'yastaff': 669,
        'yacorp': 670,  # корпоративный аккаунт для корпоративных нужд
        'test': 672,  # для проверки работоспособости ручек подписки/отписки на сид
        'vip': 671,  # указывает на аккаунты какой-нибудь известной личности
        'musicpremium': 208,  # аккаунт однажды уже был подписан на Яндекс.Музыку или пользовался ее триалом
    },
)


SERVICES = {}
sids = {}

for slug, sid in iteritems(_services):
    service = Service(slug=slug, sid=sid, is_real=slug in REAL_SERVICES)
    SERVICES[slug] = service
    sids[sid] = service

slugs = [str(k) for k in sids]
slugs.extend(SERVICES)

# Подписки, запрещенные для пдд-пользователей
_pdd_not_allowed_sids = [14, 19, 24, 61, 64, 85, 104, 119, 669]
for sid, service in iteritems(sids):
    service.for_pdd = sid not in _pdd_not_allowed_sids

# Подписки, которые позволяют обновлять свои поля
_sids_renewable = [2, 27, 42, 44]
for sid in _sids_renewable:
    sids[sid].is_renewable = True

# Подписки, хранящие в поле login не uid пользователя
_sids_stores_not_uid_in_login = [2, 16, 33, 58, 61, 68, 669]
for sid in _sids_stores_not_uid_in_login:
    sids[sid].stores_not_uid_in_login = True

# Для подписки у пользователя должен быть логин
_sids_requires_login = [2, 5, 14, 16, 19, 24, 27, 39, 59, 85, 116]
for sid in _sids_requires_login:
    sids[sid].requires_login = True

# Для подписки у пользователя должен быть пароль
_sids_requires_password = [19, 27, 59, 67, 116, 117, 119]
for sid in _sids_requires_password:
    sids[sid].requires_password = True

# Подписки, позволяющие подписывать на них
# только сервер-сервер вызовом
_sids_internal_subscribe = [
    36, 39, 42, 50, 55,
    57, 58, 59, 60, 64, 67, 70, 72, 73, 74,
    77, 78, 79, 80, 81, 83, 84, 85,
    86, 87, 88, 93, 95, 96, 97, 98, 100,
    102, 104, 106, 108, 109, 110, 111, 112,
    113, 114, 116, 117, 119, 121, 122, 123, 124, 125,
    201, 202, 203, 204, 205, 206, 207, 208, 666, 667,
    668, 669, 670, 671, 672,
]
for sid in _sids_internal_subscribe:
    sids[sid].internal_subscribe = True

# Эти сиды нужны только для существования соответствующего from.
# Подписать на них нельзя.
IMPOSSIBLE_TO_SUBSCRIBE_SIDS = [65, 70, 74, 79, 82]

# Блокирующие подписки - при удалении аккаунта с любой подпиской из этого списка, аккаунт блокируется
# Приведено в соответствие перловому Паспорту
BLOCKING_SIDS = {14, 19, 24, 64, 78, 85, 104, 116, 117, 118, 119, 125, 670}
# Для ПДД набор блокирующих сидов свой
BLOCKING_SIDS_PDD = BLOCKING_SIDS - {78}

# Некоторые альтернативные домены имеют соответствующий SID
ALT_DOMAIN_ID_TO_SID = {
    1: 61,  # galatasaray.net
}

_PROTECTED_SUBSCRIPTION_SLUGS = (
    'passport',

    'site_search',
    'slova',
    'partner',
    'public',
    'musicpartner',
    'mobilmail',

    'wdgt',
    'social',
    'cloud',
    'newspartner',
    'galatasaray',
    'strongpwd',

    'tune',
    'yandsearch',
    'mobilecloud',
    'mobilemusic',
    'contest',
    'cloudsoft',
    'cloudweb',
    'publicdemo',
    'gamification',

    'changepasswordonnextlogon',
    'pddeula',
    'pddadmin',

    'mailtest',
    'betatest',
    'yastaff',
    'yacorp',
    'test',
    'vip',
    'musicpremium',
)
_PROTECTED_SIDS = (
    69, 105,
)
PROTECTED_TO_UNSUBSCRIBE_SIDS = tuple(SERVICES[slug].sid for slug in _PROTECTED_SUBSCRIPTION_SLUGS) + _PROTECTED_SIDS

# Если на подписке установлено значение login_rule меньше либо равно указанного здесь значения, то отписка невозможна
# Значения перенесены из passport-perl
PROTECTED_SID_TO_LOGIN_RULE = {
    4: 3,
    44: 3,
    46: 3,
}


def get_service(sid=None, slug=None):
    if sid:
        service = sids.get(sid)
    else:
        service = SERVICES.get(slug)
    if service:
        # Необходимо "копировать" сервисы затем, что если
        # поменять какой-либо из сервисных атрибутов -
        # изменения происходят в наших диктах.
        return Service.from_service(service)
    else:
        # Не все сервисы есть в наших диктах.
        return Service(sid=sid, slug=slug)
