# -*- coding: utf-8 -*-
from datetime import (
    date,
    datetime,
)

from passport.backend.core.builders.mail_apis.husky import drop_mailbox
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.models.account import (
    MAIL_STATUS_ACTIVE,
    MAIL_STATUS_FROZEN,
    MAIL_STATUS_NONE,
)
from passport.backend.core.models.alias import (
    AltDomainAlias,
    YandexoidAlias,
)
from passport.backend.core.models.subscription import (
    Host,
    subscribe,
    Subscription,
    unsubscribe,
)
from passport.backend.core.services import (
    BLOCKING_SIDS,
    BLOCKING_SIDS_PDD,
    IMPOSSIBLE_TO_SUBSCRIBE_SIDS,
    PROTECTED_SID_TO_LOGIN_RULE,
    PROTECTED_TO_UNSUBSCRIBE_SIDS,
)
from passport.backend.core.undefined import Undefined


class SubscriptionError(BaseCoreError):
    """Общая ошибка при подписки пользователя на сервис"""


class SubscriptionImpossibleError(SubscriptionError):
    """Такого сида просто нет, он виртуален"""


class SubscriptionNotAllowedError(SubscriptionError):
    """Подписка на сервис запрещена для пользователей"""


class SubscriptionRequiresAccountWithLoginError(SubscriptionError):
    """Подписка на сервис требует наличие яндексового алиаса"""


class SubscriptionRequiresAccountWithPasswordError(SubscriptionError):
    """Подписка на сервис требует наличие пароля у пользователя"""


class UnsubscribeBlockingServiceError(SubscriptionError):
    """Попытка отписаться от Блокирующего сервиса"""


class UnsubscribeProtectedServiceError(SubscriptionError):
    """Попытка отписаться от сервиса, который не подразумевает отписку"""


def get_blocking_sids(account):
    if account.is_pdd:
        return BLOCKING_SIDS_PDD
    return BLOCKING_SIDS


def can_be_subscribed(account, service):
    """
    ПДД, лайт и социальных пользователей можно подписывать на ограниченный список сервисов.
    Ниже идёт несколько проверок для таких пользователей:
    - разрешение подписывать ПДД пользователей на данный сервис
    - наличие яндексового алиаса у пользователя
    - наличие пароля у пользователя
    """
    if service.sid in IMPOSSIBLE_TO_SUBSCRIBE_SIDS:
        raise SubscriptionImpossibleError()
    if service.slug == 'mail' and account.mail_status == MAIL_STATUS_FROZEN:
        # Почтовый ящик уже существует, но заморожен. Нельзя создать новый.
        raise SubscriptionImpossibleError()

    if account.is_pdd and not service.for_pdd:
        raise SubscriptionNotAllowedError()

    if service.requires_login and ((not account.login) or account.is_social or account.is_lite):
        raise SubscriptionRequiresAccountWithLoginError()

    if service.requires_password and (not account.have_password):
        if service.slug == 'cloud' and account.is_federal:
            return
        raise SubscriptionRequiresAccountWithPasswordError()


def can_be_unsubscribed(account, service):
    """
    Проверим, возможно ли отписать аккаунт от этого сервиса
    - Нельзя отписать от Блокирующих сервисов
      см. https://wiki.yandex-team.ru/passport/sids/
    - Нельзя отписать от некоторых Защищенных подписок
    - ПДД пользователь не может отписаться от Почты
    - Портальный с ПДД алиасом не может отписаться от ПочтыПро
    """

    if service.sid in get_blocking_sids(account):
        raise UnsubscribeBlockingServiceError()

    is_impossible = (
        service.sid in PROTECTED_TO_UNSUBSCRIBE_SIDS or
        ((account.is_pdd or account.is_federal) and service.slug == 'mail') or
        (account.is_normal and account.pdd_alias and service.slug == 'mailpro')
    )
    if is_impossible:
        raise UnsubscribeProtectedServiceError()


def is_subscription_blocked(subscription):
    """
    Проверяется при отписке от сервиса
    Отвечает на вопрос:
        Была ли подписка заблокирована через указание особого значения параметра login_rule?
    """
    return (
        subscription.login_rule and
        subscription.sid in PROTECTED_SID_TO_LOGIN_RULE and
        subscription.login_rule <= PROTECTED_SID_TO_LOGIN_RULE[subscription.sid]
    )


def is_subscription_blocked_by_zero(subscription):
    """
    Проверяется при подписке/отписке от сервиса
    Подписка может быть заблокирована через установленный в 0 login_rule
    Не у всех подписок есть login_rule, поэтому смотрим на login_rule
    только если он и правда есть
    """
    return subscription.login_rule is not Undefined and not subscription.login_rule


def add_subscription(account, service, login_rule=None, login=None, host_id=None):
    if service.sid == 669:
        account.yandexoid_alias = YandexoidAlias(account, login=login)
        return

    # Для intranet такого алиаса домена нет
    if service.sid == 61 and not settings.ALT_DOMAINS.get('galatasaray.net'):
        return

    if service.sid == 61:
        account.altdomain_alias = AltDomainAlias(account, login=account.login + '@galatasaray.net')
        return

    subscription = Subscription(
        parent=account,
        service=service,
        created_date=date.today(),
    )

    if login_rule is not None:
        subscription.login_rule = login_rule
    if login is not None:
        subscription.login = login

    host = Host()
    if host_id is not None:
        host.id = host_id
    elif service.sid == 2:  # Если почта
        # Поддерживаем логику по переносу данных с 16 сида ('narodmail') на 2 сид ('mail')
        if 16 in account.subscriptions:
            subscription.suid = account.subscriptions[16].suid
    if host.id:
        subscription.host = host

    subscribe(account, subscription)

    if service.slug == 'mail':
        account.mail_status = MAIL_STATUS_ACTIVE

    if service.sid == 67 and not account.totp_secret.is_set:  # strongpwd without 2fa
        account.global_logout_datetime = datetime.now()


def update_subscription(account, service, login_rule=None, host_id=None):
    subscription = account.subscriptions[service.sid]
    host = subscription.host

    if login_rule is not None:
        subscription.login_rule = login_rule
    if host_id is not None:
        host.id = host_id


def delete_subscription(account, service):
    """
    PASSP-8985 Разрешено удаление блокирующих сидов в режимах работы
    с подпиской на сервис, то есть через admsubscribe и /1/account/<uid>/subscription/.
    PASSP-13506 sid=passport считаем исключением
    PASSP-8995 — молча не отписываем от galatasaray

    :type account: Account
    :type service: Service
    :raises: ValueError
    """

    if service.sid == 8:
        raise ValueError('Unable to unsubscribe from sid=passport')
    if service.sid == 669:
        account.yandexoid_alias = None
        return

    if service.sid == 2:
        drop_mailbox(account.uid)
        account.mail_status = MAIL_STATUS_NONE

    if service.sid != 61:
        unsubscribe(account, service.sid)
    # Если почта
    if service.sid == 2 and 16 in account.subscriptions:
        unsubscribe(account, 16)


def user_has_contract_with_yandex(account):
    all_blocking_sids = get_blocking_sids(account)
    return any(s in all_blocking_sids for s in account.subscriptions)


__all__ = (
    'can_be_subscribed',
    'can_be_unsubscribed',
    'add_subscription',
    'update_subscription',
    'delete_subscription',
    'get_blocking_sids',
    'is_subscription_blocked',
    'is_subscription_blocked_by_zero',
    'user_has_contract_with_yandex',
    'SubscriptionError',
    'SubscriptionImpossibleError',
    'SubscriptionNotAllowedError',
    'SubscriptionRequiresAccountWithLoginError',
    'SubscriptionRequiresAccountWithPasswordError',
    'UnsubscribeBlockingServiceError',
    'UnsubscribeProtectedServiceError',
)
