# -*- coding: utf-8 -*-
from collections import (
    defaultdict,
    namedtuple,
)
from itertools import product
import re

from passport.backend.core import validators
from passport.backend.core.conf import settings
from passport.backend.core.suggest.first_names import get_first_names
from passport.backend.core.suggest.transliterations import get_transliteration_rules
from passport.backend.core.utils.max_heap import MaxHeap
from six import iteritems
from six.moves import xrange


TransliterationSource = namedtuple('TransliterationSource', ['string', 'weight', 'rules'])


class LoginSuggester(object):

    def __init__(self, first_name=None, last_name=None, login=None, language=None):
        self.first_name = self.clean_name(first_name) if first_name else u''
        self.last_name = self.clean(last_name) if last_name else u''
        self.login = self.clean_login(login) if login else u''
        self.lang = self.set_language(language)

        self.gender = self.identify_gender()

        # подготовим источники и правила транслитерации
        self.sources = self.generate_transliteration_sources_and_rules()

        # подготовим элементы комбинаций на основе всех собранных данных
        self.elements = self.generate_elements()

        # подготовим сами комбинации элементов
        self.mixes = self.generate_mixes()

    @property
    def name_synonyms(self):
        return get_first_names().get_name_synonyms(
            self.gender,
            self.lang,
            self.first_name,
        )

    @property
    def name_synonyms_en(self):
        return get_first_names().get_name_synonyms_en(
            self.gender,
            self.lang,
            self.first_name,
        )

    @property
    def name_manual_chunks(self):
        return get_first_names().get_name_chunks(
            self.gender,
            self.lang,
            self.first_name,
        )

    @property
    def name_auto_chunks(self):
        return get_first_names().get_auto_chunks(
            self.gender,
            self.first_name,
        )

    @property
    def transliterations(self):
        return get_transliteration_rules()

    @property
    def mixes_conf(self):
        return settings.LANG_TO_MIXES.get(self.lang, [])

    @classmethod
    def set_language(cls, language):
        if language not in settings.SUGGEST_SUPPORTED_LANGUAGES:
            return 'ru'
        return language

    @classmethod
    def clean(cls, name, cut_to=None, invalid_chars=None):
        """
        Обрезает переданную строку, убирает лишние
        символы приближенно к валидатору логина

        :type name: unicode
        :type cut_to: int
        :type invalid_chars: pattern
        :rtype: unicode
        """
        cut_to = cut_to or settings.MAX_NAME_LENGTH
        invalid_chars = invalid_chars or settings.EXCLUDE_NAME_REGEXP
        name = name[:cut_to].strip().lower()
        # Оставим только валидные буквы
        name = re.sub(invalid_chars, u'', name)
        # Убираем двойные точки
        name = re.sub(u'\.\.+', u'-', name)
        # Убираем двойные минусы
        name = re.sub(u'--+', u'-', name)
        # Убираем точки и минусы подряд
        name = re.sub(u'(\.)(-)\1?|(-)(\.)\1?', u'-', name)
        # Убераем минусы и точки в начале и в конце
        name = name.strip(u'-.')
        return name

    @classmethod
    def clean_login(cls, name):
        name = cls.clean(name, cut_to=settings.MAX_LOGIN_LENGTH, invalid_chars=settings.EXCLUDE_LOGIN_REGEXP)
        # А также убираем цифры в начале логина.
        name = name.lstrip(u'0123456789.-')
        return name

    @classmethod
    def clean_name(cls, name):
        name = cls.clean(name)
        name = name.replace(u'ё', u'е')
        return name

    def identify_gender(self):
        # Сначала определяем по фамилии
        gender = None
        male_re = settings.LANG_TO_MALE_SURNAME_REGEXP.get(self.lang)
        female_re = settings.LANG_TO_FEMALE_SURNAME_REGEXP.get(self.lang)
        if self.last_name:
            if male_re and male_re.search(self.last_name):
                gender = settings.MALE_GENDER
            if not gender and female_re and female_re.search(self.last_name):
                gender = settings.FEMALE_GENDER

        # Потом по списку имен для каждого пола
        if not gender and self.first_name:
            if self.first_name in get_first_names().names['m'].get(self.lang, {}):
                gender = settings.MALE_GENDER
            if not gender and self.first_name in get_first_names().names['f'].get(self.lang, {}):
                gender = settings.FEMALE_GENDER
        return gender or settings.UNDEFINED_GENDER

    def generate_name_synonyms(self):
        result = []
        for synonym in self.name_synonyms:
            result.append(
                TransliterationSource(
                    string=synonym,
                    weight=settings.SYNONYM_INITIAL_WEIGHT,
                    rules=self.transliterations.collect_rules(synonym, self.lang),
                ),
            )
        return result

    def generate_name_auto_chunks(self):
        auto_chunks = []
        for i, chunk in enumerate(self.name_auto_chunks):
            weight = settings.AUTO_CHUNKS_INITIAL_WEIGHT - i * settings.AUTO_CHUNKS_WEIGHT_STEP
            auto_chunks.append(
                TransliterationSource(
                    string=chunk,
                    weight=weight,
                    rules=self.transliterations.collect_rules(chunk, self.lang),
                ),
            )
        return auto_chunks

    def get_login_and_number(self):
        login = self.login
        number = None
        login_with_number = re.search(settings.LOGIN_ENDS_WITH_NUMBER_REGEXP, self.login)
        if login_with_number:
            number = login_with_number.group()
            login = re.sub(settings.LOGIN_ENDS_WITH_NUMBER_REGEXP, '', self.login)
        return login, number

    def generate_transliteration_sources_and_rules(self):
        """
        Подготовим источники и правила транслитерации.
        В источники входят следующие строки:
            * имя;
            * синонимы имени;
            * автоматические сокращения имени;
            * фамилия.
        :rtype: dict
        """
        sources = defaultdict(list)
        if self.first_name:
            sources['name_synonym'].append(
                TransliterationSource(
                    string=self.first_name,
                    weight=settings.INITIAL_WEIGHT,
                    rules=self.transliterations.collect_rules(self.first_name, self.lang),
                ),
            )
            sources['name_synonym'].extend(self.generate_name_synonyms())
            # авто-сокращения имени имеют смысл, если не определены
            # специально заготовленные сокращения в конфиге
            if not self.name_manual_chunks:
                sources['name_chunk'].extend(self.generate_name_auto_chunks())

        if self.last_name:
            sources['surname_synonym'].append(
                TransliterationSource(
                    string=self.last_name,
                    weight=settings.INITIAL_WEIGHT,
                    rules=self.transliterations.collect_rules(self.last_name, self.lang),
                ),
            )
        return sources

    def generate_elements(self):
        # TODO: штрафовать за длину строки, чтобы короткие были выше
        """
        Элементы комбинаций включают в себя:
            * логин как есть;
            * логин без номера, если пользователь ввел номер на конце;
            * логин с урезанным номером, если пользователь ввел номер вида 19хх или 20хх;
            * специально заготовленные сокращения имен (они изначально латинские);
            * специально заготовленные английские синонимы имен;
            * транслитерированные источники;
            * все что нагенерили с заменой букв на цифры;
            * префиксы.
        :rtype: dict
        """
        elements = defaultdict(MaxHeap)
        if self.login:
            elements['login_synonym'].push(settings.INITIAL_WEIGHT, self.login)
            login, number = self.get_login_and_number()
            # Если логин заканчивается на числа, добавим источник логина без числа.
            if number is not None:
                elements['login_wo_number'].push(settings.INITIAL_WEIGHT, login)
                # Если логин заканчивается на 19хх, 20хх, то добавим источник с хх только.
                year_number = re.match(settings.RECENT_YEAR_REGEXP, number)
                if year_number:
                    elements['login_number'].push(
                        settings.INITIAL_WEIGHT,
                        u'%s%s' % (login, year_number.group()[-2:]),
                    )
        # ручные чанки также всегда латинские, поэтому сразу попадают сюда
        for i, chunk in enumerate(self.name_manual_chunks):
            # Вес каждого последующего чанка уменьшается на шаг
            weight = settings.MANUAL_CHUNKS_INITIAL_WEIGHT - i * settings.MANUAL_CHUNKS_WEIGHT_STEP
            elements['name_chunk'].push(weight, chunk)

        # английские синонимы от толокеров
        for synonym in self.name_synonyms_en:
            elements['name_synonym_en'].push(settings.SYNONYM_INITIAL_WEIGHT, synonym)

        # Применяем все правила транслитерирования
        for source_type, sources in iteritems(self.sources):
            for source in sources:
                applicable_rules = source.rules
                # Если правил у источника не нашлось - добавляем источник как есть
                if not applicable_rules:
                    elements[source_type].push(source.weight, source.string)
                else:
                    for trans in self.transliterations.apply_rules(applicable_rules):
                        # На случай, если транслитерация оказалась пустая
                        if not trans['transliteration']:
                            continue
                        elements[source_type].push(
                            round(trans['factor'] * source.weight, 2),
                            trans['transliteration'],
                        )
        # Добавляем элементы с заменой букв на цифры
        for source_type, combination_elements in iteritems(elements):
            num_elements = MaxHeap()
            for comb_weight, comb_str in combination_elements.nlargest(settings.LETTER_TO_NUMBER_COMBINATION_LIMIT):
                # Все равно выкинем по длине валидатором
                if len(comb_str) >= validators.LOGIN_MAX_LENGTH:
                    continue
                if not set(comb_str).intersection(settings.LETTER_TO_NUMBER_REPLACEMENTS_KEYS):
                    continue
                applicable_rules = self.transliterations.collect_rules(
                    word=comb_str,
                    rules_source=settings.LETTER_TO_NUMBER_REPLACEMENTS,
                )
                elements_with_numbers = self.transliterations.apply_rules(
                    applicable_rules,
                    ignore_as_is=True,
                    threshold=settings.LETTER_TO_NUMBER_TRANSLITERATION_LIMIT,
                )
                for num_el in elements_with_numbers:
                    num_elements.push(
                        round(num_el['factor'] * comb_weight, 2),
                        num_el['transliteration'],
                    )
            elements[source_type].merge(num_elements)

        # Добавляем префиксы
        for prefix in settings.LOGIN_PREFIXES:
            elements['prefix'].push(settings.PREFIX_WEIGHT, prefix)
        return elements

    def generate_mixes(self):
        """
        Подготовим источники для миксов.
        Положим сюда все, что нагенерили, и провалидируем,
        чтобы потом вытаскивать в пределах лимитов
        Возвращаем список вида: {'string': str, 'weight': float}
        """
        mixes_sources = []
        for mix in self.mixes_conf:
            params = mix['params']
            params_heap = MaxHeap()
            if len(params) == 1:
                param = params[0]
                values = self.elements.get(param, MaxHeap()).nlargest()
                if not values:
                    continue
                for weight, value in values:
                    params_heap.push(round(weight * mix['factor'], 2), value)
            elif len(params) == 2:
                validator = mix['validator']
                separator = settings.SEPARATOR if mix['separator'] else ''
                param1, param2 = params
                values1 = self.elements.get(param1, MaxHeap()).nlargest()
                values2 = self.elements.get(param2, MaxHeap()).nlargest()
                if not (values1 and values2):
                    continue
                # все комбинации из существующих параметров
                params_combinations = product(values1, values2)
                for i, combination in enumerate(params_combinations):
                    string1 = combination[0][1]
                    string2 = combination[1][1]
                    if validator and string1 == string2:
                        continue
                    string = '%s%s%s' % (string1, separator, string2)
                    weight = round((combination[0][0] * combination[1][0] / 100.0) * mix['factor'], 2)
                    params_heap.push(weight, string)
                    if i >= settings.MAX_COMBINATIONS - 1:
                        break
            mixes_sources.append({
                'values': params_heap,
                'limit': mix['limit'],
            })
        return mixes_sources

    def next_pack(self):
        """
        Возвращает набор всех доступных комбинаций логинов
        :rtype: list
        """
        suggestions = []

        mixes_length = len(self.mixes)

        for i in xrange(mixes_length):
            if not self.mixes[i]['values']:
                continue
            for j in xrange(self.mixes[i]['limit']):
                try:
                    login = self.mixes[i]['values'].pop()
                    validators.Login().to_python(login)
                except (IndexError, validators.Invalid):
                    continue
                if login not in suggestions:
                    suggestions.append(login)
                if len(suggestions) >= settings.PACK_SIZE:
                    return suggestions
        return suggestions
