# -*- coding: utf-8 -*-
"""
Почанковый загрузчик записей с сохранением состояния на случай рестарта
"""
import os

from passport.backend.utils.common import chop
import six


class ChunkLoader(object):
    def __init__(
        self,
        reader,
        state_file_path=None,
        load_state=False,
        chunk_size=100,
        save_state_chunk_threshold=1,
    ):
        """
        Лоадер чанков

        :param reader: читатель
        :param state_file_path: путь к файлу состояния. Если None - сохранение не работает
        :param load_state: загрузить ли состояние при старте
        :param chunk_size: размер генерируемых чанков
        :param save_state_chunk_threshold: каждые Х чанков сохранять состояние
        """
        self.chunks_since_last_save = 0
        self.reader = reader

        self.state_file_path = state_file_path
        self.save_state_chunk_threshold = save_state_chunk_threshold
        if load_state:
            self._load_state()
        else:
            self._assert_state_empty()

        self.rec_iterator = chop(reader, chunk_size)

    @property
    def pos(self):
        return self.reader.pos

    def _load_state(self):
        if not os.path.exists(self.state_file_path):
            raise RuntimeError('State file doesn\'t exist')
        with open(self.state_file_path) as f:
            self.reader.seek(int(f.readline()))

    def _assert_state_empty(self):
        if os.path.exists(self.state_file_path):
            raise RuntimeError('State file already exists')

    def _save_state_if_needed(self):
        if self.chunks_since_last_save >= self.save_state_chunk_threshold:
            with open(self.state_file_path, 'w') as f:
                f.write(str(self.pos))
            self.chunks_since_last_save = 0

    def __iter__(self):
        return self

    def __next__(self):
        res = six.next(self.rec_iterator)
        self.chunks_since_last_save += 1
        self._save_state_if_needed()
        return res

    next = __next__
