# -*- coding: utf-8 -*-

"""
Ридеры для почанкового лоадера
"""


class AbstractReader(object):
    @property
    def pos(self):
        raise NotImplementedError()

    def seek(self, pos):
        raise NotImplementedError()

    def __iter__(self):
        return self

    def __next__(self):
        raise NotImplementedError()

    next = __next__


class FileReader(AbstractReader):
    """
    Читатель из файлов. Надо открывать через context manager или .open
    Обрезает newline
    """

    def __init__(self, path):
        self._pos = 0
        self.path = path
        self.file_obj = None

    @property
    def pos(self):
        return self._pos

    def seek(self, pos):
        if self.file_obj is None:
            raise RuntimeError('File reader is closed')
        while self.pos < pos:
            try:
                next(self)
            except StopIteration:
                raise ValueError('File line position {} is out of bounds'.format(pos))

    def open(self):
        self.file_obj = open(self.path)

    def close(self):
        self.file_obj.close()

    def __next__(self):
        res = next(self.file_obj).rstrip('\n')
        self._pos += 1
        return res

    next = __next__

    def __enter__(self):
        self.open()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()
        self.file_obj = None
