# -*- coding: utf-8 -*-
from collections import defaultdict

import mock
from passport.backend.core.conf import settings
from passport.backend.core.host.host import get_current_host
from passport.backend.core.redis_manager import redis_manager
from passport.backend.core.redis_manager.faker.fake_redis import FakeRedis
from passport.backend.core.redis_manager.redis_manager import get_redis_mgr
from passport.backend.core.test.test_utils import single_entrant_patch
from passport.backend.core.tracks.track_manager import TrackManager


@single_entrant_patch
class FakeTrackManager(object):
    def __init__(self, db_config=None):
        self.config = db_config or {'master': {'host': 'localhost', 'port': 6379}}
        self._redis_patch = mock.patch('redis.Redis', FakeRedis)

    def start(self):
        self._redis_patch.start()
        for redis_name, config in settings.REDIS_CONFIG.items():
            get_redis_mgr(redis_name).configure(self.config)

    def stop(self):
        self._redis_patch.stop()
        redis_manager._redises = defaultdict(redis_manager.RedisManager)

    def create_test_track(self, manager=None, track_type='register', is_short=False, process_name=None):
        if manager is None:
            manager = self.get_manager()
        if not is_short:
            track = manager.create(track_type, 'dev', process_name=process_name)
        else:
            track = manager.create_short(track_type, 'dev', process_name=process_name)
        self.existing_track_id = track.track_id
        return self.existing_track_id

    def get_manager(self):
        return TrackManager()

    def get_manager_and_trackid(self, track_type_to_create='register', is_short=False, process_name=None):
        manager = self.get_manager()
        track_id = self.create_test_track(manager, track_type_to_create, is_short=is_short, process_name=process_name)
        return manager, track_id

    def get_redis(self):
        return get_redis_mgr(get_current_host().get_id())


@single_entrant_patch
class FakeTrackIdGenerator(object):
    def __init__(self):
        self._mock = mock.Mock()
        self._create_track_id_patch = mock.patch('passport.backend.core.tracks.utils._create_track_id', self._mock)

    def start(self):
        self._create_track_id_patch.start()
        return self

    def stop(self):
        self._create_track_id_patch.stop()

    def set_return_value(self, value):
        self.set_side_effect(None)
        self._mock.return_value = value

    def set_side_effect(self, effect):
        self._mock.side_effect = effect
