# -*- coding: utf-8 -*-
from copy import deepcopy
import json
import logging

import six


class TrackList(object):
    """Список: обёртка над элементом словаря _lists родителя"""
    def __init__(self, parent, list_name):
        self.parent = parent
        self.list_name = list_name

    def append(self, *args):
        self.parent._lists.setdefault(self.list_name, []).extend(args)

    def get(self):
        return deepcopy(self.parent._lists.get(self.list_name, []))


class TrackCounter(object):
    """Счётчик: обёртка над элементом словаря _counters родителя"""
    def __init__(self, parent, counter_name):
        self.parent = parent
        self.counter_name = counter_name

    def get(self, default=None):
        value = self.parent._counters.get(self.counter_name)
        return int(value) if value is not None else default

    def incr(self):
        if self.counter_name not in self.parent._counters:
            self.parent._counters[self.counter_name] = 0
        self.parent._counters[self.counter_name] += 1
        return self.parent._counters[self.counter_name]

    def reset(self):
        self.parent._counters[self.counter_name] = 0


class TrackFieldBase(object):
    def __init__(self, name=None, is_sensitive=False, allow_edit_concurrently=True):
        if name is not None:
            logging.warning(
                '"name" attribute is set automatically, and overriding it will break '
                'process of loading track contents from external data. You\'ve been warned.',
            )

        self.name = name
        self.is_sensitive = is_sensitive
        self.allow_edit_concurrently = allow_edit_concurrently

    def __get__(self, obj, objtype):
        raise AttributeError()

    def __set__(self, obj, value):
        raise AttributeError()


class TrackReadOnlyField(TrackFieldBase):
    def __get__(self, obj, objtype):
        if obj is None:
            return self
        return obj._data.get(self.name)


class TrackField(TrackFieldBase):
    def __get__(self, obj, objtype):
        if obj is None:
            return self
        return obj._data.get(self.name)

    def __set__(self, obj, value):
        if (
            value is not None and
            (
                not isinstance(value, (six.text_type, six.binary_type, float) + six.integer_types) or
                isinstance(value, bool)
            )
        ):
            raise ValueError('Can\'t write %s to %s (TrackField)' % (type(value), self.name))
        obj._data[self.name] = value


class TrackFlagField(TrackFieldBase):

    def __get__(self, obj, objtype):
        if obj is None:
            return self
        value = obj._data.get(self.name)
        if value is None:
            return None
        else:
            return value == '1'

    def __set__(self, obj, value):
        if value is None:
            obj._data.pop(self.name, None)
            return
        obj._data[self.name] = '1' if value else '0'


class TrackJsonSerializableField(TrackFieldBase):

    def __get__(self, obj, objtype):
        if obj is None:
            return self
        value = obj._data.get(self.name)
        if value is None:
            return None
        return json.loads(value)

    def __set__(self, obj, value):
        obj._data[self.name] = json.dumps(value)


class TrackCounterField(TrackFieldBase):

    def __get__(self, obj, objtype):
        if obj is None:
            return self
        return TrackCounter(obj, self.name)

    def __set__(self, obj, value):
        if isinstance(value, TrackCounter):
            if value.get() is not None:
                obj._counters[self.name] = value.get()
            else:
                obj._counters.pop(self.name, None)
        else:
            raise TypeError('Unable to set value %s (%s) to %s (TrackCounterField)' % (
                value,
                type(value).__name__,
                self.name,
            ))


class TrackListField(TrackFieldBase):

    def __get__(self, obj, objtype):
        if obj is None:
            return self
        return TrackList(obj, self.name)

    def __set__(self, obj, value):
        if isinstance(value, TrackList):
            obj._lists[self.name] = value.get()
        else:
            raise TypeError('Unable to set value %s (%s) to %s (TrackListField)' % (
                value,
                type(value).__name__,
                self.name,
            ))


class TrackUnixtimeField(TrackFieldBase):
    """Поле для хранения времени в формате unixtime"""

    def __get__(self, obj, objtype):
        """Ожидается на чтение строка"""
        if obj is None:
            return self
        value = obj._data.get(self.name)
        if value is None:
            return

        return float(value)

    def __set__(self, obj, value):
        """Ожидается на запись результат вызова time.time()"""
        if value is None:
            raise ValueError('None value can\'t be written to %s (TrackUnixtimeField)' % self.name)

        if not isinstance(value, (int, float)):
            raise ValueError('Can\'t write not a number (int or float) to %s (TrackUnixtimeField)' % self.name)

        obj._data[self.name] = str(value)
