# -*- coding: utf-8 -*-
from copy import deepcopy
import inspect
import sys

from passport.backend.core.conf import settings
from passport.backend.core.tracks.fields import (
    TrackCounter,
    TrackCounterField,
    TrackField,
    TrackFieldBase,
    TrackFlagField,
    TrackJsonSerializableField,
    TrackList,
    TrackListField,
    TrackReadOnlyField,
)
from passport.backend.utils.string import smart_text
from six import (
    add_metaclass,
    iteritems,
    string_types,
)


class TrackMeta(type):

    def __new__(mcs, name, bases, attrs):
        attrs['_meta'] = set()
        attrs['_meta_sensitive_fields'] = set()
        attrs['_meta_concurrent_protected_fields'] = set()
        attrs['_meta_lists'] = set()
        for parent in filter(lambda x: hasattr(x, '_meta'), bases):
            attrs['_meta'].update(parent._meta)
            attrs['_meta_sensitive_fields'].update(parent._meta_sensitive_fields)
            attrs['_meta_concurrent_protected_fields'].update(parent._meta_concurrent_protected_fields)
            attrs['_meta_lists'].update(parent._meta_lists)
        for obj_name in attrs.keys():
            field = attrs[obj_name]

            # Заполняем список секретных полей
            if isinstance(field, TrackFieldBase):
                attrs['_meta'].add(obj_name)
                if field.is_sensitive:
                    attrs['_meta_sensitive_fields'].add(obj_name)
                if not field.allow_edit_concurrently:
                    attrs['_meta_concurrent_protected_fields'].add(obj_name)
                if isinstance(field, TrackListField):
                    attrs['_meta_lists'].add(obj_name)

                if field.name is None:
                    field.name = obj_name
        return type.__new__(mcs, name, bases, attrs)


@add_metaclass(TrackMeta)
class TrackMixin(object):
    pass


@add_metaclass(TrackMeta)
class TrackBase(object):
    track_type = TrackReadOnlyField()
    created = TrackReadOnlyField()

    # Имя процесса, выполняемого с использованием данного трека
    process_name = TrackField()

    track_id = None
    track_version = None

    def __init__(self, track_id, data=None, lists=None, ttl=None, version=None):
        """
        Создаёт объект трека
          track_id - id трека
          data - словарь полей трека (включая счётчики)
          lists - словарь списков
          ttl - время жизни трека
        """
        self.track_id = track_id
        self.track_version = int(version or 0)

        self._data = dict(data) if data else {}
        self._counters = {}
        self._lists = {}
        self._ttl = ttl or settings.TRACK_TTL

        for name in self._meta:
            field = getattr(self, name)
            if isinstance(field, TrackList):
                value = (lists or {}).get(name, [])
                self._lists[name] = value
            elif isinstance(field, TrackCounter):
                value = int(self._data.pop(name, 0))
                if value:
                    self._counters[name] = value
            elif isinstance(field, string_types):
                if name in self._data:
                    self._data[name] = smart_text(self._data[name])

    @property
    def ttl(self):
        return self._ttl

    @property
    def sensitive_fields(self):
        return self._meta_sensitive_fields

    @property
    def concurrent_protected_fields(self):
        """
        Поля, значения которых нельзя изменять в concurrent-апдейтах
        """
        return self._meta_concurrent_protected_fields

    @property
    def list_names(self):
        return self._meta_lists

    @property
    def logout_checkpoint_timestamp(self):
        """
        Свойство отдает таймстемп, с которым нужно сравнивать
        таймстемпы глогаута и отзыва всех веб-сессий для определения валидности трека
        """
        result = getattr(self, 'password_verification_passed_at', None)
        return float(result or self.created)

    def snapshot(self):
        return self.__class__(
            self.track_id,
            data=dict(self._data, **self._counters),
            lists=deepcopy(self._lists),
            version=self.track_version,
        )

    def __repr__(self):
        return '<%s: %s>' % (self.__class__.__name__, self.track_id)

    def __setattr__(self, name, value):
        if not hasattr(self, name) and not name.startswith('_'):
            raise AttributeError('%s has no field "%s"' % (self.__class__.__name__, name))
        super(TrackBase, self).__setattr__(name, value)

    # data - словарь <имя поля на модели трека>: <значение поля>
    def parse(self, data):
        for key, new_value in iteritems(data):
            field = getattr(self, key)
            if isinstance(field, (TrackList, TrackCounter)):
                raise AttributeError('Can not assign to field "%s". Wrong type: %s' % (key, type(field)))
            setattr(self, key, new_value)


class CaptchaTrackMixin(TrackMixin):
    """Поля, заполняемые при прохождении капчи"""

    bruteforce_status = TrackField()
    is_captcha_checked = TrackFlagField()
    is_captcha_required = TrackFlagField()
    is_captcha_recognized = TrackFlagField()
    is_captcha_ignored = TrackFlagField()
    captcha_generate_count = TrackCounterField()
    captcha_check_count = TrackCounterField()
    captcha_generated_at = TrackField()
    captcha_checked_at = TrackField()
    socialreg_captcha_required_count = TrackCounterField()
    image_captcha_type = TrackField()
    voice_captcha_type = TrackField()
    captcha_key = TrackField()
    captcha_image_url = TrackField()
    captcha_voice_url = TrackField()
    captcha_voice_intro_url = TrackField()


class PasswordTrackMixin(TrackMixin):
    is_captcha_checked = TrackFlagField()
    is_captcha_required = TrackFlagField()
    is_captcha_recognized = TrackFlagField()


class PhoneConfirmationTrackMixin(TrackMixin):
    """Знание о подтвержденном телефонном номере пользователя и поля для валидации телефонов"""

    phone_confirmation_phone_number = TrackField()
    phone_confirmation_phone_number_original = TrackField()
    # FrodoInfo зависит от этого поля
    phone_confirmation_first_send_at = TrackField()
    phone_confirmation_last_send_at = TrackField()
    # TODO Удалить после выкатки PASSP-32458 (через 3 часа)
    phone_confirmation_sms = TrackField(is_sensitive=True)
    phone_confirmation_code = TrackField(is_sensitive=True)
    phone_confirmation_first_checked = TrackField()
    phone_confirmation_last_checked = TrackField()
    phone_confirmation_is_confirmed = TrackFlagField()
    phone_confirmation_confirms_count = TrackCounterField()
    phone_confirmation_sms_count = TrackCounterField()
    return_masked_number = TrackFlagField()

    sanitize_phone_changed_phone = TrackFlagField()
    sanitize_phone_error = TrackFlagField()
    sanitize_phone_count = TrackCounterField()
    sanitize_phone_first_call = TrackField()
    sanitize_phone_last_call = TrackField()

    phone_confirmation_send_ip_limit_reached = TrackFlagField()
    phone_confirmation_send_count_limit_reached = TrackFlagField()
    phone_confirmation_confirms_count_limit_reached = TrackFlagField()

    phone_valid_for_call = TrackFlagField()
    phone_valid_for_flash_call = TrackFlagField()
    phone_validated_for_call = TrackField()
    phone_confirmation_method = TrackField()
    phone_confirmation_first_called_at = TrackField()
    phone_confirmation_last_called_at = TrackField()
    phone_confirmation_calls_count = TrackCounterField()
    phone_confirmation_calls_count_limit_reached = TrackFlagField()
    phone_confirmation_calls_ip_limit_reached = TrackFlagField()
    phone_call_session_id = TrackField()
    phone_confirmation_used_gate_ids = TrackField()

    phone_operation_confirmations = TrackListField()

    enable_phonenumber_alias_as_email = TrackFlagField()

    def copy_phone_confirmation_from_other_track(self, other_track):
        """
        Копирует поля класса PhoneConfirmationTrackMixin из other_track в этот
        трек.

        Значение поля типа TrackListField скопируется, только если значение
        списка в self является приставкой для соответствующего значения
        в other_track. Так работает из-за того что у списков треков реализована
        только операция append.
        """
        phone_field_names = set()
        for attr_name in dir(PhoneConfirmationTrackMixin):
            attr = getattr(PhoneConfirmationTrackMixin, attr_name)
            if isinstance(attr, TrackFieldBase):
                phone_field_names.add(attr_name)

        for field_name in phone_field_names:
            new_value = getattr(other_track, field_name)
            setattr(self, field_name, new_value)


class ConfirmationCodeTrackMixin(TrackMixin):
    """Поля для хранения данных при ожидании подтверждения лайта"""
    confirmation_code = TrackField()


class DeviceInfoTrackMixin(TrackMixin):
    """Информация об устройстве и процессе на нем (от мобильных и десктопных клиентов)"""

    account_manager_version = TrackField()

    device_language_sys = TrackField()
    device_locale = TrackField()
    device_geo_coarse = TrackField()
    device_hardware_id = TrackField()  # Уникальный идентификатор телефонного аппарата (из Метрики)
    device_manufacturer = TrackField()  # Производитель аппарата
    device_hardware_model = TrackField()  # Модель телефона
    device_os_id = TrackField()  # Операционная система телефона (она же Платформа)
    device_os_version = TrackField()  # Версия операционной системы телефона
    device_application = TrackField()  # Название мобильного приложения, записавшего данные (на Android может
    # содержать версию)
    device_application_version = TrackField()  # Версия мобильного приложения
    device_app_uuid = TrackField()  # Уникальный идентификатор мобильного приложения
    device_cell_provider = TrackField()
    device_clid = TrackField()
    device_ifv = TrackField()
    device_id = TrackField()  # Уникальный идентификатор устройства (произвольный)
    device_name = TrackField()  # Имя устройства, например, "Ivan's iPhone"

    cloud_token = TrackField()  # Токен из облачного хранилища

    scenario = TrackField()  # Имя сценария АМ - записывается вместе с другими полями
    avatar_size = TrackField()  # желаемый размер аватарки (зависит от размера экрана девайса)
    captcha_scale_factor = TrackField()  # желаемый масштаб капчи (зависит от размера экрана девайса)

    gps_package_name = TrackField()  # Имя андроид-приложения (для sms retriever)

    # Выставляется, если клиент, пришедший в API, считается фиктивным: неизвестен настоящий ip, неизвестен настоящий
    # user agent, даже хоста может не быть.
    is_fake_client = TrackFlagField()


class RequestInfoMixin(TrackMixin):
    """Общая информация о параметрах пришедшего запроса"""

    language = TrackField()
    display_language = TrackField()
    country = TrackField()
    retpath = TrackField()
    origin = TrackField()
    service = TrackField()
    process_uuid = TrackField()
    ysa_mirror_resolution = TrackField()
    js_fingerprint = TrackField()
    session_reissue_interval = TrackField()  # PASSP-38825 временно добавлено

    csrf_token = TrackField(is_sensitive=True)

    surface = TrackField()

    browser_id = TrackField()
    os_family_id = TrackField()
    region_id = TrackField()


class AccountInfoMixin(TrackMixin):
    """Информация об аккаунте пользователя. Отражает информацию из модели пользователя"""

    uid = TrackField(allow_edit_concurrently=False)
    login = TrackField()
    domain = TrackField()
    human_readable_login = TrackField()  # account.human_readable_login
    machine_readable_login = TrackField()  # account.machine_readable_login
    have_password = TrackFlagField()
    password_hash = TrackField(is_sensitive=True)
    emails = TrackField()
    birthday = TrackField()

    # В процессе авторизации поле следует ставить в True, если это парольная авторизация.
    is_password_passed = TrackFlagField()

    # Будет ли у пользователя политика "крепкого пароля"
    is_strong_password_policy_required = TrackFlagField()

    # TODO: Определиться с местом для этого поля. Используется:
    # passport_api.common.authorization
    # passport_api.views.oauth
    # passport_api.views.password
    # passport_api.views.complete
    # passport_api.views.register - только тут записывается False, если пользователь uncompleted
    allow_authorization = TrackFlagField()
    allow_oauth_authorization = TrackFlagField()

    # Дополнительная причина, по которой происходит авторизация (см. authtypes.py)
    auth_source = TrackField()

    # Тип авторизационного челленжа, который прошёл пользователь
    auth_challenge_type = TrackField()

    # Отвечает на вопрос: есть ли уже у пользователя защищенный номер для sms
    # Для фронта. Определяет нужно ли привязывать защищенный телефон пользователю или отправить sms на тот что есть.
    # Используется при смене пароля.
    has_secure_phone_number = TrackFlagField()

    # Отвечает на вопрос: есть ли уже у пользователя защищенный номер.
    # Определяет нужно ли валидировать пароль пользователя с учётом телефона.
    can_use_secure_number_for_password_validation = TrackFlagField()

    # Поле отвечающее за сохранение в трэке защищённого номера пользователя в e164
    secure_phone_number = TrackField()

    additional_data_asked = TrackField()


class TotpMixin(TrackMixin):
    # пин пользователя, который "замешивается" с app_secret для генерации totp-пароля.
    totp_pin = TrackField(is_sensitive=True)
    # app_secret, в формате base32. Именно его хранит приложение Яндекс.Ключ
    totp_app_secret = TrackField(is_sensitive=True)

    # сериализованные пин и секрет(-ы) для otp-авторизации, в закодированном виде, которые вернул ЧЯ
    # нужны при включении 2fa
    totp_secret_encrypted = TrackField(is_sensitive=True)

    # 2фа-секреты пользователя (словарь: id -> время добавления)
    totp_secret_ids = TrackJsonSerializableField()

    # "Картинки", используемые Ключом в качестве дополнительного фактора при входе по 2фа
    correct_2fa_picture = TrackField(is_sensitive=True)
    correct_2fa_picture_expires_at = TrackField()

    # список device_id на которых настроен Ключ
    totp_push_device_ids = TrackListField()

    # crsf секрет для добавления девайсов в список выше
    push_setup_secret = TrackField(is_sensitive=True)


class WebauthnMixin(TrackMixin):
    """Поля для работы с webauthn"""
    # случайная соль для прохождения webauthn-авторизации
    webauthn_challenge = TrackField()
    # id секрета, владение которым подтвердил пользователь
    webauthn_confirmed_secret_external_id = TrackField()


class SessionTrackMixin(TrackMixin):
    """Знание о текущей и прошлой сессии пользователя"""

    session = TrackField(is_sensitive=True)
    sslsession = TrackField(is_sensitive=True)
    session_created_at = TrackField()
    sessguard = TrackField(is_sensitive=True)

    # Предыдущая сессия
    old_session = TrackField(is_sensitive=True)
    # Предыдущая защищенная сессия
    old_ssl_session = TrackField(is_sensitive=True)
    # ttl старой валидной сессии
    old_session_ttl = TrackField()
    # время создания сессии
    old_session_create_timestamp = TrackField()
    # IP, с которого была создана сессия
    old_session_ip = TrackField()
    # возраст сессии
    old_session_age = TrackField()
    # Предыдущая кука sessguard
    old_sessguard = TrackField(is_sensitive=True)

    # Вид сессии: sessional | permanent
    authorization_session_policy = TrackField()
    # Нужно ли ограничить срок жизни данной сессии
    is_session_restricted = TrackFlagField()

    # Не менять дефолт при добавлении аккаунта в сессию
    dont_change_default_uid = TrackFlagField()

    # Время последнего успешного ввода пароля
    password_verification_passed_at = TrackField()

    # Идентификатор авторизации (не меняется при обмене одного креденшла на другой)
    login_id = TrackField()

    # ID сессии пользователя, пробросившего свою авторизацию на другое устройство
    source_authid = TrackField()

    # Значения счётчиков badauth из последнего запроса в метод login ЧЯ
    badauth_counts = TrackJsonSerializableField()


class CookiesMixin(TrackMixin):
    """Знание о неавторизационных куках пользователя"""
    cookie_l_value = TrackField()
    cookie_my_value = TrackField()
    cookie_yp_value = TrackField()
    cookie_ys_value = TrackField()
    cookie_yandex_login_value = TrackField()
    cookie_yandex_gid_value = TrackField()


class UserInputMixin(TrackMixin):
    # Логин, который пользователь ввел при авторизации
    user_entered_login = TrackField()
    user_entered_email = TrackField()


class SocialRegistrationTrackMixin(TrackMixin):
    social_task_data = TrackJsonSerializableField()
    social_task_id = TrackField()
    social_place = TrackField()
    social_return_brief_profile = TrackFlagField()
    # Режим выхлопа процесса социальной авторизации: вид креденшла, который выдадим в конце процесса
    social_output_mode = TrackField()
    # Начавший социальную авторизацию потребитель
    social_broker_consumer = TrackField()
    # значения, необходимые для защиты выдачи креденшла (например, кода в OAuth) с помощью PKCE
    oauth_code_challenge = TrackField()
    oauth_code_challenge_method = TrackField()
    social_track_id = TrackField()
    # Идентификатор зарегистрированного социальщика
    social_register_uid = TrackField()


class OAuthClientInfoMixin(TrackMixin):
    x_token_client_id = TrackField()
    x_token_client_secret = TrackField()
    client_id = TrackField()
    client_secret = TrackField()


class OAuthTrackMixin(TrackMixin):
    # Создавался ли для трека oauth-token
    is_oauth_token_created = TrackFlagField()  # TODO: выпилить через 3 часа после выкатки PASSP-19048 в прод
    oauth_token_created_at = TrackField()

    payment_auth_retpath = TrackField()  # урл для возврата после прохождения платёжной авторизации


class TrackStateMixin(TrackMixin):
    """Флаги сообщают, что некоторый процесс прошёл успешно и закончен"""

    # Нужно ли завершить авторизацию установкой пароля
    # для автозарегистрированного пользователя
    is_complete_autoregistered = TrackFlagField()
    # дорегистрация ПДД пользователя без задания пароля
    is_complete_pdd = TrackFlagField()
    # дорегистрация ПДД пользователя c заданием пароля
    is_complete_pdd_with_password = TrackFlagField()
    # принудительная дорегистрация lite пользователя
    is_force_complete_lite = TrackFlagField()
    # Нужно ли завершить авторизацию сменой пароля
    is_password_change = TrackFlagField()
    # Происходит ли сейчас авторизация по oauth-token'у
    is_oauth_pdd_authorization = TrackFlagField()
    # Требуется ли ввод второго фактора (например, одноразового пароля)
    is_second_step_required = TrackFlagField()
    # Список возможных вторых факторов
    allowed_second_steps = TrackJsonSerializableField()

    # Флаг сообщает, что с данным трэком успешно зарегистрировали пользователя
    is_successful_registered = TrackFlagField()
    # Флаг сообщает, что с данным трэком пользователь успешно был дорегистрирован
    is_successful_completed = TrackFlagField()
    # Флаг сообщает, что с данным трэком пользователь успешно подтвердил/привязал телефон/создал алиас
    # Используется только в телефонных бандлах
    is_successful_phone_passed = TrackFlagField()

    is_auth_challenge_shown = TrackFlagField()
    is_avatar_secret_checked = TrackFlagField()
    do_not_save_fresh_profile = TrackFlagField()

    # пользователь уже был на ручке /callback, то есть вернулся из брокера
    social_is_callbacked = TrackFlagField()

    # Отвечает на вопрос: Идет ли сейчас регистрация с привязкой телефона? Влияет на счетчики СМС
    is_it_registration_with_phone = TrackFlagField()

    # Отвечает на вопрос: Идет ли сейчас процесс включения двухфакторной аутентификации
    is_it_otp_enable = TrackFlagField()

    # Отвечает на вопрос: Идет ли сейчас процесс выключения двухфакторной аутентификации
    is_it_otp_disable = TrackFlagField()

    # Отвечает на вопрос: Идет ли сейчас процесс восстановления аккаунта с включенной 2FA.
    is_it_otp_restore = TrackFlagField()

    # Отвечает на вопрос: прошло ли успешно восстановление 2FA
    is_otp_restore_passed = TrackFlagField()

    # Флаг, показывающий нужна ли проверка пользователя
    # с помощью валидации телефона при смене пароля
    is_change_password_sms_validation_required = TrackFlagField()

    # Была ли проведена успешная проверка PIN?
    is_pin_checked = TrackFlagField()

    # Проставляется вместе с is_password_change на принудительной смене.
    is_force_change_password = TrackFlagField()

    # Используется, чтобы записать факт завершения проверки отп
    # на включениии/выключении двухфакторной аутентификации
    is_otp_checked = TrackFlagField()

    # Отвечает на вопрос: Идет ли сейчас процесс смены аватара
    is_avatar_change = TrackFlagField()

    # Отвечает на вопрос: были ли отозваны веб-сессии для данного трека (явно или глогаутом)
    is_web_sessions_logout = TrackFlagField()

    # Отвечает на вопрос: был ли пользователь успешно отписан от сервиса?
    is_unsubscribed = TrackFlagField()

    # Проведена авторизация через OTP-магию
    is_otp_magic_passed = TrackFlagField()

    # Проведена авторизация через X-TOKEN-магию
    is_x_token_magic_passed = TrackFlagField()

    # Пройдена аутентификация по одноразовому ключу
    is_key_auth_passed = TrackFlagField()

    # Не выставлять неавторизационные куки заново, а взять значения из трека
    use_non_auth_cookies_from_track = TrackFlagField()

    # FIXME: После рефакторинга состояний трэков, надо убить это поле
    # Используется в:
    # * views.bundle.phone.base.BasePhoneBundleSubmitter
    # * views.bundle.phone.base.BasePhoneBundleCommitter
    state = TrackField()

    # Поле для хранения фронтендом состояния какого-либо процесса
    frontend_state = TrackField()

    # ID трека для следующей операции
    next_track_id = TrackField()

    # ID связанного persistent-трека
    persistent_track_id = TrackField()


class SuggestAndValidateMixin(TrackMixin):
    """Информация о ходе заполнения регистрационной формы"""

    # FrodoInfo зависит от этого поля
    suggested_logins = TrackListField()
    suggest_login_count = TrackCounterField()
    suggest_login_first_call = TrackField()
    suggest_login_last_call = TrackField()

    suggest_name_count = TrackCounterField()
    suggest_language_count = TrackCounterField()
    suggest_gender_count = TrackCounterField()
    suggest_country_count = TrackCounterField()
    suggest_timezone_count = TrackCounterField()
    phone_number_validation_count = TrackCounterField()
    control_questions_count = TrackCounterField()

    login_validation_count = TrackCounterField()
    login_validation_first_call = TrackField()
    login_validation_last_call = TrackField()

    password_validation_count = TrackCounterField()
    password_validation_first_call = TrackField()
    password_validation_last_call = TrackField()

    hint_validation_count = TrackCounterField()
    retpath_validation_count = TrackCounterField()

    # Количество попыток ввода неправильного отп
    invalid_otp_count = TrackCounterField()


class ConfirmationCountersMixin(TrackMixin):
    # количество попыток ввода неверного ключа при подтверждении email
    invalid_email_key_count = TrackCounterField()
    # количество попыток угадать ответ на КВ
    answer_checks_count = TrackCounterField()


class AutoRestoreBaseMixin(TrackMixin):
    """Информация о процессе восстановления"""
    restore_state = TrackField()
    current_restore_method = TrackField()
    # Для составных методов восстановления - последный шаг восстановления,
    # выполненный пользователем в рамках метода восстановления
    last_restore_method_step = TrackField()
    restore_methods_select_counters = TrackJsonSerializableField()  # Счетчики переключений между методами восстановления
    restore_methods_select_order = TrackListField()  # Список выбранных методов восстановления
    # Для восстановления по email-адресу
    email_checks_count = TrackCounterField()  # Число неуспешных проверок email-адреса
    is_email_check_passed = TrackFlagField()  # Признак успешного ввода email-адреса, подходящего для восстановления
    restoration_emails_count = TrackCounterField()  # Число отправленных сообщений с кодом восстановления
    restoration_key_created_at = TrackField()  # Timestamp создания ключа восстановления
    # Для восстановления 2ФА с использованием короткой анкеты
    restore_2fa_form_checks_count = TrackCounterField()
    # Для восстановления с использованием ссылки от саппортов
    support_link_type = TrackField()
    # Для показа промо привязки после окончания восстановления
    has_restorable_email = TrackFlagField()


class AutoRestoreUserInputMixin(TrackMixin):
    """Информация о введенных пользователем данных при автоматическом восстановлении доступа"""

    user_entered_question_id = TrackField()
    user_entered_question = TrackField()
    user_entered_answer = TrackField(is_sensitive=True)
    user_entered_phone_number = TrackField()
    user_entered_firstname = TrackField()
    user_entered_lastname = TrackField()


class SemiAutoRestoreMixin(TrackMixin):
    """Информация, специфичная для полуавтоматического восстановления"""
    request_source = TrackField()
    semi_auto_step = TrackField()
    version = TrackField()
    questions = TrackJsonSerializableField()
    factors = TrackJsonSerializableField(is_sensitive=True)
    events_info_cache = TrackJsonSerializableField()
    is_for_learning = TrackFlagField()  # Анкета используется для обучения
    is_unconditional_pass = TrackFlagField()  # Анкета используется для пропуска пользователя в саппорт без проверок


class OtpRestoreMixin(TrackMixin):
    """Информация, специфичная для восстановления при включённой 2fa"""
    secure_phone_entered = TrackFlagField()
    failed_pins = TrackJsonSerializableField()
    secure_phone_checks_count = TrackCounterField()
    pin_check_errors_count = TrackCounterField()


class RobotLikenessMixin(TrackMixin):
    """Информация для принятия решения о роботности пользователя"""
    page_loading_info = TrackField()
    check_css_load = TrackFlagField()
    check_js_load = TrackFlagField()


class MigrationTrackMixin(TrackMixin):
    """
    Информация, которая нужна для миграции аккаунтов.
    """
    allow_session_authorization_for_migration = TrackFlagField()
    allow_oauth_authorization_for_migration = TrackFlagField()


class EmailConfirmationMixin(TrackMixin):
    """
    Информация для подтверждения адреса электронной почты.
    """
    email_confirmation_address = TrackField(allow_edit_concurrently=False)
    email_confirmation_code = TrackField()
    email_confirmation_checks_count = TrackCounterField()
    email_confirmation_passed_at = TrackField()


class EmailCheckOwnershipMixin(TrackMixin):
    """
    Информация для подтверждения владения адресом электронной почты.
    Отличается от EmailConfirmationMixin тем, что подтверждает любой
    привязанный email.
    """
    email_check_ownership_passed = TrackFlagField()
    email_check_ownership_code = TrackField()
    email_ownership_checks_count = TrackCounterField()


class MagicLinkMixin(TrackMixin):
    """
    Информация для авторизации и регистрации методом "магических ссылок"
    """
    magic_link_secret = TrackField()
    magic_link_code = TrackField()
    magic_link_send_to = TrackField()
    magic_link_sent_time = TrackField()
    magic_link_start_time = TrackField()
    magic_link_start_browser = TrackField()
    magic_link_start_location = TrackField()
    magic_link_confirms_count = TrackCounterField()
    magic_link_confirm_time = TrackField()
    magic_link_invalidate_time = TrackField()
    magic_link_message_id = TrackField()
    require_auth_for_magic_link_confirm = TrackFlagField()


class AuthMixin(TrackMixin):
    """
    Разные поля для авторизации
    """
    # Поля для МДА
    fretpath = TrackField()
    clean = TrackField()

    # Причина редиректа на смену пароля
    change_password_reason = TrackField()

    # Кэш ответа ручки /submit
    submit_response_cache = TrackJsonSerializableField()

    # Разрешённые способы авторизации
    allowed_auth_methods = TrackJsonSerializableField()
    # Используемый метод авторизации
    auth_method = TrackField()

    # Значение otp
    otp = TrackField(is_sensitive=True)
    # Разрешена авторизация через OTP-магию
    is_allow_otp_magic = TrackFlagField()
    # Логин, с которым разрешено проводить магическую авторизацию
    login_required_for_magic = TrackField()
    # Успешность проверки x-token для магической авторизации из АМ
    # FIXME: выпилить в пользу cred_status через 3 часа после выкладки PASSP-33412 в прод
    x_token_status = TrackField()
    # Успешность проверки x-token или куки для магической авторизации из АМ
    cred_status = TrackField()
    # для телевизора и кода для входа
    magic_qr_device_code = TrackField()

    # Поле используется в процессе принудительной смены пароля
    # для проверки успешности ответа на КВКО (проверяем вопросы из истории)
    is_fuzzy_hint_answer_checked = TrackFlagField()

    # Поле используется в процессе отписки от сервиса
    # для проверки успешности ответа на текущий КВ/КО
    is_secure_hint_answer_checked = TrackFlagField()

    # Поле используется в процессе принудительной смены пароля
    # для проверки успешности обработки короткой анкеты (ФИО/ДР, дата и место
    # регистрации, IP).
    is_short_form_factors_checked = TrackFlagField()

    # Поле для кеширования login_status при вызове method=login ЧЯ
    blackbox_login_status = TrackField()
    # Поле для кеширования password_status при вызове method=login ЧЯ
    blackbox_password_status = TrackField(is_sensitive=True)
    # Поле для кеширования totp_check_time при вызове method=login ЧЯ
    blackbox_totp_check_time = TrackField()

    # Поле для кеширования тэгов ответа antifraud/score ручки. Необходимо для прокидывания в ручку челленджей значений тэгов
    antifraud_tags = TrackJsonSerializableField()
    antifraud_external_id = TrackField()

    # Признак, что пользователь логиниться в сервисе, которые умеет школьников
    allow_scholar = TrackFlagField()

    # Область действия выписываемой сессии. Например, сессия должна действовать
    # только на школьных сервисах.
    session_scope = TrackField()


class Push2faMixin(TrackMixin):
    """
    Поля для челленджей через пуш
    """
    push_otp = TrackField()
    allow_set_xtoken_trusted = TrackFlagField()


class Challenge3DSMixin(TrackMixin):
    """
    Поля для челленджей через 3ds
    """
    payment_status = TrackField()
    paymethod_id = TrackField()
    purchase_token = TrackField()


# === Типы треков ===

class UniversalTrack(AccountInfoMixin,
                     AuthMixin,
                     AutoRestoreBaseMixin,
                     AutoRestoreUserInputMixin,
                     CaptchaTrackMixin,
                     Challenge3DSMixin,
                     ConfirmationCodeTrackMixin,
                     ConfirmationCountersMixin,
                     CookiesMixin,
                     DeviceInfoTrackMixin,
                     EmailCheckOwnershipMixin,
                     EmailConfirmationMixin,
                     MagicLinkMixin,
                     MigrationTrackMixin,
                     OAuthClientInfoMixin,
                     OAuthTrackMixin,
                     OtpRestoreMixin,
                     PhoneConfirmationTrackMixin,
                     Push2faMixin,
                     RequestInfoMixin,
                     RobotLikenessMixin,
                     SemiAutoRestoreMixin,
                     SessionTrackMixin,
                     SocialRegistrationTrackMixin,
                     SuggestAndValidateMixin,
                     TotpMixin,
                     TrackStateMixin,
                     UserInputMixin,
                     WebauthnMixin,
                     TrackBase):
    """Трек со всеми возможными полями"""
    # FIXME: Перекрывает доступ к readonly полю трека
    track_type = 'universal'


class AuthTrack(UniversalTrack):
    """Трек для авторизации пользователя"""

    track_type = 'authorize'


class RegisterTrack(UniversalTrack):
    """Трек для регистрации нового пользователя"""

    track_type = 'register'


class CompleteTrack(UniversalTrack):
    """Трек для завершения(дополнения) регистрации пользователя"""

    track_type = 'complete'


class RestoreTrack(UniversalTrack):
    """Трек для выполнения операций восстановления доступа"""

    track_type = 'restore'


track_classes = inspect.getmembers(
    sys.modules[__name__],
    lambda obj: inspect.isclass(obj) and issubclass(obj, TrackBase) and obj is not TrackBase,
)
TRACK_TYPES = {cls.track_type: cls for cls_name, cls in track_classes}


__all__ = (
    'TrackBase',
    'AuthTrack',
    'RegisterTrack',
    'CompleteTrack',
    'RestoreTrack',
)
