# -*- coding: utf-8 -*-
import logging
from time import time

from passport.backend.core.conf import settings
from passport.backend.core.redis_manager.redis_manager import get_redis_mgrs
from passport.backend.core.tracks.exceptions import NodeNotFoundError
from passport.backend.core.tracks.reader import read
from passport.backend.core.tracks.serializer import Serializer
from passport.backend.core.tracks.transaction import TrackTransaction
from passport.backend.core.tracks.utils import (
    create_short_track_id,
    create_track_id,
    get_model_by_type,
    get_node_id_from_track_id,
)


log = logging.getLogger('passport.tracks.TrackManager')


class TrackManager(object):

    def __init__(self, redis_managers=None):
        self.redis_track_managers = redis_managers or get_redis_mgrs()

    def _get_redis_node(self, track_id):
        node_key = get_node_id_from_track_id(track_id)
        node = self.redis_track_managers.get(node_key)
        if not node:
            raise NodeNotFoundError('node not found for key: %s' % node_key)
        return node

    def read(self, track_id):
        return read(
            track_id=track_id,
            redis_node=self._get_redis_node(track_id),
            ttl_offset=settings.TRACK_TTL_OFFSET,
        )

    def _create(self, track_id, track_type, consumer, created=None, ttl=None, process_name=None):
        model_class = get_model_by_type(track_type)

        track = model_class(
            track_id=track_id,
            data={
                'track_type': track_type,
                'process_name': process_name,
                'consumer': consumer,  # FIXME: а поля-то такого в треке нет
                'created': created or time(),
            },
            ttl=ttl,
            version=1,
        )
        Serializer.execute(
            old_track=None,
            new_track=track,
            redis_node=self._get_redis_node(track_id),
        )

        log.info('Created track: %s', track_id)
        return track

    def create(
        self,
        track_type,
        consumer,
        created=None,
        ttl=None,
        process_name=None,
        track_id=None,
    ):
        if track_id is None:
            track_id = create_track_id()
        return self._create(track_id, track_type, consumer, created, ttl, process_name)

    def create_short(
        self,
        track_type,
        consumer,
        created=None,
        ttl=None,
        process_name=None,
        track_id=None,
    ):
        if track_id is None:
            track_id = create_short_track_id()
        return self._create(track_id, track_type, consumer, created, ttl, process_name)

    def transaction(self, track_id=None, track=None, allow_nested=False):
        track = track or self.read(track_id)
        return TrackTransaction(track, self, allow_nested=allow_nested)
