# -*- coding: utf-8 -*-
from os import urandom
import random

from passport.backend.core.conf import settings
from passport.backend.core.host.host import get_current_host
from passport.backend.core.tracks.exceptions import InvalidTrackIdError
from passport.backend.core.tracks.model import TRACK_TYPES
from passport.backend.utils.common import bytes_to_hex
from passport.backend.utils.string import (
    smart_bytes,
    smart_text,
)
import six


def get_model_by_type(track_type):
    track_model = TRACK_TYPES.get(track_type)
    if track_model is None:
        raise ValueError('Wrong track_type: "%s"' % track_type)
    return track_model


def create_track_id():
    return _create_track_id()


def _create_track_id():
    """
    "Обычный" ID, 16 случайных байт кодируются в hex-представлении,
    в конце дописывается ID хоста
    """
    rnd_id = urandom(settings.TRACK_RANDOM_BYTES_COUNT)
    host_id = chr(get_current_host().get_id())
    if six.PY2:
        track_id = (rnd_id + host_id).encode('hex')
    else:
        host_id = host_id.encode('latin1')
        track_id = (rnd_id + host_id).hex()
    return track_id


SHORT_TRACK_ALPHA = '123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz'


def create_short_track_id():
    """
    Короткий ID, представление в алфавите base58
    """
    length = settings.SHORT_TRACK_LENGTH - 2
    track_id = ''
    sys_random = random.SystemRandom()
    for _ in range(length):
        track_id += sys_random.choice(SHORT_TRACK_ALPHA)
    host_id = chr(get_current_host().get_id())
    track_id += bytes_to_hex(smart_bytes(host_id))
    return track_id


def get_node_id_from_track_id(track_id):
    return int(track_id[-2:], 16)


def check_track_id(track_id):
    track_id = smart_text(track_id)
    error_message = u'track id "%s" does not seem to be a real track id' % track_id
    if len(track_id) not in settings.ALLOWED_TRACK_LENGTHS:
        raise InvalidTrackIdError(error_message)
    try:
        get_node_id_from_track_id(track_id)
    except ValueError:
        raise InvalidTrackIdError(error_message)


def make_redis_key(key):
    return 'track:%s' % key


def make_redis_subkey(track_id, subkey_name):
    return '%s:%s' % (make_redis_key(track_id), subkey_name)
