# -*- coding: utf-8 -*-
from passport.backend.core.conf import settings
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.core.turboapp_partners.schemas import (
    allowed_hosts_table as aht,
    allowed_urls_table as aut,
)
from passport.backend.core.ydb.declarative import select
from passport.backend.core.ydb.declarative.elements import and_
from passport.backend.core.ydb.exceptions import YdbNoResultFound
from passport.backend.core.ydb.ydb import (
    Ydb,
    YdbQueryExecutor,
)


@lazy_loadable()
class YdbTurboappPartners(Ydb):
    def __init__(self, endpoint=None, database=None, auth_token=None, use_tvm=None, graphite_logger=None, enabled=None):
        super(YdbTurboappPartners, self).__init__(
            endpoint=endpoint or settings.YDB_ENDPOINT,
            database=database or settings.YDB_TURBOAPP_PARTNERS_DATABASE,
            enabled=enabled if enabled is not None else settings.YDB_TURBOAPP_PARTNERS_ENABLED,
            use_tvm=use_tvm if use_tvm is not None else settings.YDB_USE_TVM,
            auth_token=auth_token or settings.YDB_TOKEN,
            graphite_logger=graphite_logger,
        )


def get_ydb_turboapp_partners():
    return LazyLoader.get_instance('YdbTurboappPartners')


def _get_executor():
    return YdbQueryExecutor(get_ydb_turboapp_partners())


def _run_query_and_get_one(query):
    ydb_executor = _get_executor()
    query_c = query.compile()
    result = ydb_executor.execute_queries([query_c])[0]
    try:
        return result.one()
    except YdbNoResultFound:
        return


def get_partner_id_by_host(host):
    query = select(aht, aht.c.host == host)
    row = _run_query_and_get_one(query)
    return row['partner_id'] if row else None


def get_partner_id_by_url(url):
    query = select(aut, aut.c.url == url)
    row = _run_query_and_get_one(query)
    return row['partner_id'] if row else None


def is_psuid_allowed(host):
    query = select(
        aht,
        and_(aht.c.host == host, aht.c.allow_psuid),
    )
    row = _run_query_and_get_one(query)
    return bool(row)
