# -*- coding: utf-8 -*-

import mock
from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.core.test.test_utils import (
    NotInitializedMock,
    single_entrant_patch,
)
from passport.backend.core.tvm.tvm_credentials_manager import TvmCredentialsManager
from passport.backend.utils.common import noneless_dict
import ticket_parser2
import ticket_parser2.low_level
from ticket_parser2.unittest import create_user_ticket_for_unittest


TEST_KEYS = (
    '1:CpgCCpMCCAEQABqIAjCCAQQCggEAcLEXeH67FQESFUn4_7wnX7wN0PUrBoUsm3QQ4W5vC-qz6sXaEjSwnTV8w1o-z6X9KPL'
    'lhzMQvuS38NCNfK4uvJ4Zvfp3YsXJ25-rYtbnrYJHNvHohD-kPCCw_yZpMp21JdWigzQGuV7CtrxUhF-NNrsnUaJrE5-OpEWN'
    't4X6nCItKIYeVcSK6XJUbEWbrNCRbvkSc4ak2ymFeMuHYJVjxh4eQbk7_ZPzodP0WvF6eUYrYeb42imVEOR8ofVLQWE5DVnb1'
    'z_TqZm4i1XkS7jMwZuBxBRw8DGdYei0lT_sAf7KST2jC0590NySB3vsBgWEVs1OdUUWA6r-Dvx9dsOQtSCVkQYQAAqZAgqUAg'
    'gCEAAaiQIwggEFAoIBAQDhEBM5-6YsPWfogKtbluJoCX1WV2KdzOaQ0-OlRbBzeCzw-eQKu12c8WakHBbeCMd1I1TU64SDkDo'
    'rWjXGIa_2xT6N3zzNAE50roTbPCcmeQrps26woTYfYIuqDdoxYKZNr0lvNLLW47vBr7EKqo1S4KSj7aXK_XYeEvUgIgf3nVIc'
    'Nrio7VTnFmGGVQCepaL1Hi1gN4yIXjVZ06PBPZ-DxSRu6xOGbFrfKMJeMPs7KOyE-26Q3xOXdTIa1X-zYIucTd_bxUCL4BVbw'
    'W2AvbbFsaG7ISmVdGu0XUTmhXs1KrEfUVLRJhE4Dx99hAZXm1_HlYMUeJcMQ_oHOhV94ENFIJaRBhACCpYBCpEBCAMQABqGAT'
    'CBgwKBgF9t2YJGAJkRRFq6fWhi3m1TFW1UOE0f6ZrfYhHAkpqGlKlh0QVfeTNPpeJhi75xXzCe6oReRUm-0DbqDNhTShC7uGU'
    'v1INYnRBQWH6E-5Fc5XrbDFSuGQw2EYjNfHy_HefHJXxQKAqPvxBDKMKkHgV58WtM6rC8jRi9sdX_ig2NIJeRBhABCpYBCpEB'
    'CAQQABqGATCBgwKBgGB4d6eLGUBv-Q6EPLehC4S-yuE2HB-_rJ7WkeYwyp-xIPolPrd-PQme2utHB4ZgpXHIu_OFksDe_0bPg'
    'ZniNRSVRbl7W49DgS5Ya3kMfrYB4DnF5Fta5tn1oV6EwxYD4JONpFTenOJALPGTPawxXEfon_peiHOSBuQMu3_Vn-l1IJiRBh'
    'ADCpcBCpIBCAUQABqHATCBhAKBgQCTJMKIfmfeZpaI7Q9rnsc29gdWawK7TnpVKRHws1iY7EUlYROeVcMdAwEqVM6f8BVCKLG'
    'gzQ7Gar_uuxfUGKwqEQzoppDraw4F75J464-7D5f6_oJQuGIBHZxqbMONtLjBCXRUhQW5szBLmTQ_R3qaJb5vf-h0APZfkYhq'
    '1cTttSCZkQYQBAqWAQqRAQgLEAAahgEwgYMCgYBvvGVH_M2H8qxxv94yaDYUTWbRnJ1uiIYc59KIQlfFimMPhSS7x2tqUa2-h'
    'I55JiII0Xym6GNkwLhyc1xtWChpVuIdSnbvttbrt4weDMLHqTwNOF6qAsVKGKT1Yh8yf-qb-DSmicgvFc74mBQm_6gAY1iQsf'
    '33YX8578ClhKBWHSCVkQYQAAqXAQqSAQgMEAAahwEwgYQCgYEAkuzFcd5TJu7lYWYe2hQLFfUWIIj91BvQQLa_Thln4YtGCO8'
    'gG1KJqJm-YlmJOWQG0B7H_5RVhxUxV9KpmFnsDVkzUFKOsCBaYGXc12xPVioawUlAwp5qp3QQtZyx_se97YIoLzuLr46UkLcL'
    'nkIrp-Jo46QzYi_QHq45WTm8MQ0glpEGEAIKlwEKkgEIDRAAGocBMIGEAoGBAIUzbxOknXf_rNt17_ir8JlWvrtnCWsQd1MAn'
    'l5mgArvavDtKeBYHzi5_Ak7DHlLzuA6YE8W175FxLFKpN2hkz-l-M7ltUSd8N1BvJRhK4t6WffWfC_1wPyoAbeSN2Yb1jygtZ'
    'JQ8wGoXHcJQUXiMit3eFNyylwsJFj1gzAR4JCdIJeRBhABCpYBCpEBCA4QABqGATCBgwKBgFMcbEpl9ukVR6AO_R6sMyiU11I'
    '8b8MBSUCEC15iKsrVO8v_m47_TRRjWPYtQ9eZ7o1ocNJHaGUU7qqInFqtFaVnIceP6NmCsXhjs3MLrWPS8IRAy4Zf4FKmGOx3'
    'N9O2vemjUygZ9vUiSkULdVrecinRaT8JQ5RG4bUMY04XGIwFIJiRBhADCpYBCpEBCA8QABqGATCBgwKBgGpCkW-NR3li8GlRv'
    'qpq2YZGSIgm_PTyDI2Zwfw69grsBmPpVFW48Vw7xoMN35zcrojEpialB_uQzlpLYOvsMl634CRIuj-n1QE3-gaZTTTE8mg-AR'
    '4mcxnTKThPnRQpbuOlYAnriwiasWiQEMbGjq_HmWioYYxFo9USlklQn4-9IJmRBhAEEpUBCpIBCAYQABqHATCBhAKBgQCoZkF'
    'Gm9oLTqjeXZAq6j5S6i7K20V0lNdBBLqfmFBIRuTkYxhs4vUYnWjZrKRAd5bp6_py0csmFmpl_5Yh0b-2pdo_E5PNP7LGRzKy'
    'KSiFddyykKKzVOazH8YYldDAfE8Z5HoS9e48an5JsPg0jr-TPu34DnJq3yv2a6dqiKL9zSCakQYSlQEKkgEIEBAAGocBMIGEA'
    'oGBALhrihbf3EpjDQS2sCQHazoFgN0nBbE9eesnnFTfzQELXb2gnJU9enmV_aDqaHKjgtLIPpCgn40lHrn5k6mvH5OdedyI6c'
    'CzE-N-GFp3nAq0NDJyMe0fhtIRD__CbT0ulcvkeow65ubXWfw6dBC2gR_34rdMe_L_TGRLMWjDULbNIJqRBg'
)
TEST_CLIENT_ID = 28
TEST_CLIENT_SECRET = 'GRMJrKnj4fOVnvOqe-WyD1'

TEST_CLIENT_ID_2 = 229
TEST_CLIENT_ID_3 = 339
TEST_ALIAS = 'alias1'
TEST_TICKET = (
    '3:serv:CBAQ__________9_IhkI5QEQHBoIYmI6c2VzczEaCGJiOnNlc3My:WUPx1cTf05fjD1exB35T5j2DCHWH1YaLJon_a'
    '4rN-D7JfXHK1Ai4wM4uSfboHD9xmGQH7extqtlEk1tCTCGm5qbRVloJwWzCZBXo3zKX6i1oBYP_89WcjCNPVe1e8jwGdLsnu6'
    'PpxL5cn0xCksiStILH5UmDR6xfkJdnmMG94o8'
)
TEST_INVALID_TICKET = '3:serv:ololo'
TEST_TICKET_DATA = {
    str(TEST_CLIENT_ID_2): {
        'alias': TEST_ALIAS,
        'ticket': TEST_TICKET,
    },
}


def fake_tvm_credentials_data(keys=TEST_KEYS, client_id=TEST_CLIENT_ID, client_secret=TEST_CLIENT_SECRET,
                              ticket_data=TEST_TICKET_DATA):
    return dict(
        keys=keys,
        client_id=client_id,
        client_secret=client_secret,
        tickets=ticket_data,
    )


def fake_user_ticket(default_uid=1, scopes=None, uids=None):
    if uids is None:
        uids = [default_uid]
    args = dict(
        status='Ok',
        default_uid=default_uid,
        scopes=scopes,
        uids=uids,
    )
    args = noneless_dict(args)
    return args


def fake_invalid_user_ticket(status='Expired', default_uid=0, scopes=None, uids=None):
    args = dict(
        status=status,
        default_uid=default_uid,
        scopes=scopes,
        uids=uids or [100500],
    )
    args = noneless_dict(args)
    return args


@single_entrant_patch
class FakeTvmCredentialsManager(object):
    def __init__(self):
        self._mock = mock.Mock()
        self._patch_read_configs = mock.patch.object(
            TvmCredentialsManager,
            '_read_configs',
            self._mock,
        )

    def start(self):
        self._patch_read_configs.start()
        LazyLoader.flush(instance_name='TvmCredentialsManager')

    def stop(self):
        self._patch_read_configs.stop()

    def set_data(self, data):
        self._mock.return_value = data

    def __enter__(self):
        self.start()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.stop()


@single_entrant_patch
class FakeTvmTicketChecker(object):
    def __init__(self):
        self._check_user_ticket_side_effect = mock.Mock(side_effect=[])
        self.__patches = [
            mock.patch.object(
                FakeUserContext,
                'check',
                side_effect=self._fake_check_user_ticket,
            ),
            mock.patch(
                'passport.backend.core.tvm.tvm_credentials_manager.UserContext',
                FakeUserContext,
            ),
        ]

    def start(self):
        for patch in self.__patches:
            patch.start()

    def stop(self):
        for patch in reversed(self.__patches):
            patch.stop()

    def set_check_user_ticket_side_effect(self, fake_user_tickets):
        """
        Входным параметром может быть список словарей описывающих User Ticket.

        Например, валидный User Ticket:
            dict(
                status='Ok',
                default_uid=TEST_UID1,
                scopes=['carsharing:all'],
                uids=[TEST_UID1],
            )

        Недействительный User Ticket:
            dict(status='Expired')

        Весь список статусов можно найти в тексте ticket parser2:
        https://a.yandex-team.ru/arc/trunk/arcadia/library/python/deprecated/ticket_parser2/ticket_parser2/ticket_parser2_pymodule.pyx?rev=5177104#L110
        """
        self._check_user_ticket_side_effect.side_effect = fake_user_tickets

    def _fake_check_user_ticket(self, *args, **kwargs):
        try:
            fake_user_ticket = self._check_user_ticket_side_effect()
        except StopIteration:
            raise NotInitializedMock(type(self).__name__)
        if 'status' in fake_user_ticket:
            fake_user_ticket = dict(fake_user_ticket)
            fake_user_ticket['status'] = getattr(
                ticket_parser2.Status,
                fake_user_ticket['status'],
            )
        return create_user_ticket_for_unittest(**fake_user_ticket)


class FakeUserContext(ticket_parser2.low_level.UserContext):
    """
    Над UserContext нужна обёртка, потому что нельзя патчить классы
    c-extensions.
    """
