# -*- coding: utf-8 -*-
import logging
import os

from passport.backend.core.conf import settings
from passport.backend.core.dynamic_config import (
    BaseDynamicConfig,
    load_json,
    load_plaintext,
)
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.core.tvm.exceptions import (
    LoadTvmCredentialsError,
    UnavailableUserContextError,
    UnknownDestinationError,
)
from passport.backend.utils.common import merge_dicts
from six import (
    iteritems,
    itervalues,
)
from ticket_parser2 import BlackboxEnv
from ticket_parser2.low_level import (
    ServiceContext,
    UserContext,
)


log = logging.getLogger('passport.tvm_credentials_manager')


RESULTS_LOCATION = '/var/cache/yandex/passport-tvm-keyring/'

_BLACKBOX_NAME_TO_ENV = dict(
    production=BlackboxEnv.Prod,
    testing=BlackboxEnv.Test,
    intranet_production=BlackboxEnv.ProdYateam,
    intranet_testing=BlackboxEnv.TestYateam,
    stress=BlackboxEnv.Stress,
)


@lazy_loadable()
class TvmCredentialsManager(BaseDynamicConfig):
    def __init__(
        self,
        keyring_config_name=None,
        cache_time=None,
        ticket_parser2_blackbox_env=None,
    ):
        keyring_config_name = keyring_config_name or settings.TVM_KEYRING_CONFIG_NAME
        cache_time = cache_time or settings.TVM_CREDENTIALS_CACHE_TIME

        super(TvmCredentialsManager, self).__init__(
            config_filenames=[
                os.path.join(RESULTS_LOCATION, 'tvm.keys'),
                os.path.join(RESULTS_LOCATION, '%s.tickets' % keyring_config_name),
            ],
            cache_time=cache_time,
        )
        self._client_id = None
        self._client_secret = None
        self._keys = None
        self._service_context = None
        self._user_context = None
        self._tickets_by_client_id = {}
        self._tickets_by_alias = {}
        self._ticket_parser2_blackbox_env = ticket_parser2_blackbox_env or settings.TICKET_PARSER2_BLACKBOX_ENV

    def read_config_file(self, filename):
        try:
            if filename.endswith('.keys'):
                data = load_plaintext(filename)
                config = {'keys': data.strip()}
            else:
                config = load_json(filename)
        except ValueError as e:
            raise LoadTvmCredentialsError(str(e))
        except IOError as e:
            raise LoadTvmCredentialsError(u'{msg} ({filename})'.format(
                msg=e.args[1],
                filename=e.filename,
            ))
        log.debug(
            'TVM credentials successfully loaded from file %s',
            filename,
        )
        return config

    def merge_configs(self, configs):
        return merge_dicts(*configs)

    def postprocess(self):
        self._client_id = self.config['client_id']
        self._client_secret = self.config.get('client_secret')
        self._keys = self.config['keys']
        self._service_context = ServiceContext(self._client_id, self._client_secret, self._keys)
        self._tickets_by_client_id = {
            int(client_id): value_dict['ticket']
            for client_id, value_dict in iteritems(self.config.get('tickets', {}))
            if value_dict.get('ticket')
        }
        self._tickets_by_alias = {
            value_dict['alias']: value_dict['ticket']
            for value_dict in itervalues(self.config.get('tickets', {}))
            if value_dict.get('ticket')
        }
        self._user_context = self._build_user_context()

    def _build_user_context(self):
        blackbox_env = _BLACKBOX_NAME_TO_ENV.get(self._ticket_parser2_blackbox_env)
        if blackbox_env is None:
            log.warning('Unable to find suitable ticket parser blackbox environment: %s' % self._ticket_parser2_blackbox_env)
            return
        return UserContext(blackbox_env, self._keys)

    def get_ticket_by_client_id(self, client_id):
        self.load()
        client_id = int(client_id)
        if client_id not in self._tickets_by_client_id:
            raise UnknownDestinationError(client_id)
        return self._tickets_by_client_id[client_id]

    def get_ticket_by_alias(self, alias):
        self.load()
        if alias not in self._tickets_by_alias:
            raise UnknownDestinationError(alias)
        return self._tickets_by_alias[alias]

    def get_user_context(self):
        self.load()
        if not self._user_context:
            raise UnavailableUserContextError()
        return self._user_context

    @property
    def keys(self):
        return self._keys

    @property
    def client_id(self):
        return self._client_id

    @property
    def client_secret(self):
        return self._client_secret

    @property
    def service_context(self):
        return self._service_context


def get_tvm_credentials_manager():
    manager = LazyLoader.get_instance('TvmCredentialsManager')
    manager.load()
    return manager
