# -*- coding: utf-8 -*-

import datetime
import re

import six


BIRTHDAY_RE = re.compile('^(\d{4})-(\d{2})-(\d{2})$')


@six.python_2_unicode_compatible
class Birthday(object):

    def __init__(self, year=None, month=None, day=None):
        # Четверка здесь для правильной обработки 29 февраля. Используется только на стадии валидации,
        # само значение дальше нигде не должно участвовать.
        self._date = datetime.date(year or 4, month or 1, day or 1)
        self._set_mask = [bool(x) for x in [year, month, day]]

    @classmethod
    def parse(cls, date):
        if date is not None and BIRTHDAY_RE.match(date):
            return cls(*[int(x) for x in date.split('-')])
        else:
            raise ValueError('Wrong date format, %Y-%m-%d is required')

    @property
    def date(self):
        return self._date

    @property
    def is_year_set(self):
        return self._set_mask[0]

    @property
    def is_date_full(self):
        return all(self._set_mask)

    def __hash__(self):
        return hash(str(self))

    def __str__(self):
        values = zip([self._date.year, self._date.month, self._date.day], self._set_mask, [4, 2, 2])
        birthday = '-'.join(str(part).zfill(length) if is_set else '0' * length for part, is_set, length in values)
        return six.text_type(birthday)

    def __repr__(self):
        return '<Birthday %s>' % str(self)

    def __eq__(self, other):
        return str(self) == str(other)

    def __ne__(self, other):
        return not self.__eq__(other)


def parse_birthday(key):
    def _parse_birthday(data, *args):
        if key not in data:
            return False, None

        birth_date = data.get(key)

        if birth_date is None:
            return True, None

        if isinstance(birth_date, Birthday):
            return True, birth_date
        else:
            try:
                return True, Birthday.parse(birth_date)
            except ValueError:
                return True, None

    return _parse_birthday
