# -*- coding: utf-8 -*-


class BitVector(object):
    """
    Простая структура, которая позволяет эффективно работать с битами числа,
    как со словарем {<index>: <0 or 1>}.
    Также позволяет работать с полями по имени (как в namedtuple).
    """
    field_names = None  # словарь {имя: индекс}

    def __init__(self, data=None):
        if data is None:
            self._int_data = 0
        elif isinstance(data, int):
            self._int_data = data
        else:
            raise TypeError(type(data))

    def __int__(self):
        return self._int_data

    def __repr__(self):
        return '%s(data=%s)' % (self.__class__.__name__, self._int_data)  # pragma: no cover

    def __eq__(self, other):
        return (self.__class__ == other.__class__) and (self._int_data == other._int_data)

    def __ne__(self, other):
        return not self.__eq__(other)

    def __getattr__(self, key):
        if self.field_names is not None and key in self.field_names:
            return self[self.field_names[key]]
        raise AttributeError(key)

    def __setattr__(self, key, value):
        if self.field_names is not None and key in self.field_names:
            self[self.field_names[key]] = value
        else:
            super(BitVector, self).__setattr__(key, value)

    def __getitem__(self, index):
        return (self._int_data >> index) & 1

    def __setitem__(self, index, value):
        if isinstance(value, bool):
            value = int(value)

        if value == 0:
            self._int_data &= ~(1 << index)
        elif value == 1:
            self._int_data |= 1 << index
        else:
            raise ValueError('Invalid value for BitVector')


class PhoneOperationFlags(BitVector):
    field_names = {
        'aliasify': 0,
        'should_ignore_binding_limit': 1,
        'in_quarantine': 2,
    }


class PhoneBindingsFlags(BitVector):
    field_names = {
        'should_ignore_binding_limit': 0,
    }
