# -*- coding: utf-8 -*-
import six


FORMAT_PASSPORT = 1
FORMAT_SOCIAL = 2
FORMAT_TEMPLATE = 3

PREFIX_PASSPORT = 'p'
PREFIX_SOCIAL = 's'
PREFIX_TEMPLATE = 't'


@six.python_2_unicode_compatible
class DisplayName(object):
    def __init__(self, name=None, provider=None, profile_id=None, public_name=None):
        self._reset()
        self.name = name
        self.public_name = public_name or name

        if provider:
            self.set_social_provider(provider, profile_id)

    def _reset(self):
        self.name = None
        self.provider = None
        self.profile_id = None
        self.format = FORMAT_PASSPORT

    def set_social_provider(self, provider, profile_id):
        if provider and profile_id is None:
            raise TypeError('profile_id must be set for social profiles')
        self.provider = provider
        self.profile_id = profile_id
        self.format = FORMAT_SOCIAL

    def set(self, display_name):
        if not display_name:
            return
        self._reset()

        head, tail = display_name.split(':', 1)
        self.provider = None

        if head == PREFIX_SOCIAL:
            profile_id, provider, name = tail.split(':', 2)
            self.set_social_provider(provider, int(profile_id))
            tail = name
        elif head == PREFIX_PASSPORT:
            self.format = FORMAT_PASSPORT
        elif head == PREFIX_TEMPLATE:
            self.format = FORMAT_TEMPLATE
        else:
            raise ValueError('unknown prefix')

        self.name = self.public_name = tail

    def convert_to_passport(self):
        if not self.is_social:
            raise TypeError('Only social display_name can be converted to passport format')
        self.provider = None
        self.profile_id = None
        self.format = FORMAT_PASSPORT

    @property
    def is_passport(self):
        return self.format == FORMAT_PASSPORT

    @property
    def is_social(self):
        return self.format == FORMAT_SOCIAL

    @property
    def is_template(self):
        return self.format == FORMAT_TEMPLATE

    def __str__(self):
        if not self.name:
            return u''
        if self.is_social:
            return u'%s:%s:%s:%s' % (
                PREFIX_SOCIAL,
                self.profile_id,
                self.provider,
                self.name,
            )
        elif self.is_passport:
            return u'%s:%s' % (PREFIX_PASSPORT, self.name)
        elif self.is_template:
            return u'%s:%s' % (PREFIX_TEMPLATE, self.name)

    def __eq__(self, other):
        return six.text_type(self) == six.text_type(other)

    def __repr__(self):
        return '<DisplayName: %s>' % self

    def as_dict(self, with_public_name=False):
        display_name = dict(
            name=self.name or u'',
        )
        if with_public_name:
            display_name.update(
                public_name=self.public_name or u'',
            )
        if self.is_social:
            display_name['social'] = {
                'provider': self.provider,
                'profile_id': self.profile_id,
            }

        return display_name
