# -*- coding: utf-8 -*-

import hashlib

from passport.backend.core.conf import settings
from passport.backend.core.types.login.login import (
    normalize_login,
    raw_login_from_email,
)
from passport.backend.core.types.phone_number.phone_number import parse_phone_number
from passport.backend.utils.string import smart_unicode
from six import itervalues


class Email(object):
    # Нативные - email адреса, на которые можно посылать почту и она будет получена
    # Внешние - email адреса, которые сообщает владелец самостоятельно.
    # С внешних email адресов происходит сбор почты.
    # Нативный email адрес всегда является подтверждённым.

    # 20.09.2013 spleenjack: флаг is_prohibit_restore не нужен, использовать нигде не будем.
    # Можно ли восстанавливаться этим email адресом - определяем сами.

    def __init__(self, is_native, is_confirmed, is_rpop, is_unsafe,
                 is_default, is_silent, address, creation_datetime=None):

        # Признак внутреннего email.
        self.is_native = is_native

        # Признак подтвержденности адреса.
        self.is_confirmed = is_confirmed

        # Признак почтового ящика, письма с которого забирает почтовый сборщик Яндекса.
        # Детали использования:
        # http://wiki.yandex-team.ru/passport/blackbox#07.12.2010rpopemailattribute
        self.is_rpop = is_rpop

        # Признак адреса, который был подтвержден небезопасно.
        self.is_unsafe = is_unsafe

        # Признак того, что email является адресом по умолчанию.
        self.is_default = is_default

        # Признак того, что на данный email нельзя отправлять нотификации
        self.is_silent = is_silent

        self.creation_datetime = creation_datetime
        self.address = address
        self._username = None
        self._domain = None

    # Признак внешнего email.
    @property
    def is_external(self):
        return not self.is_native

    @property
    def is_suitable_for_restore(self):
        return (
            self.is_confirmed and
            self.is_external and
            not self.is_rpop and
            not self.is_unsafe and
            not self.is_silent
        )

    @property
    def username(self):
        if not self._username and '@' in self.address:
            self._username, self._domain = self.address.split('@')

        return self._username

    @property
    def domain(self):
        if not self._domain and '@' in self.address:
            self._username, self._domain = self.address.split('@')

        return self._domain

    def __lt__(self, other):
        if isinstance(other, Email):
            return self.address < other.address
        return False

    def __hash__(self):
        to_hash = '%s.%s.%s.%s.%s.%s' % (self.is_native, self.is_confirmed, self.is_rpop,
                                         self.is_unsafe, self.is_default, self.address)
        return int(hashlib.md5(to_hash.encode('utf-8')).hexdigest(), 16)

    def __str__(self):
        return str(self.address)

    def __repr__(self):
        return '<Email: %s>' % str(self)

    def __eq__(self, other):
        if isinstance(other, Email):
            if self.__hash__() == other.__hash__():
                return True
        return False

    def __ne__(self, other):
        return not self.__eq__(other)


class Emails(object):
    def __init__(self):
        self._emails = dict()
        self._default = None

    def add(self, email):
        self._emails[email.address] = email

        if email.is_default:
            self._default = email

    def __iter__(self):
        return itervalues(self._emails)

    @property
    def default(self):
        return self._default

    def find(self, address):
        return self._emails.get(address)

    @property
    def native(self):
        return [email for email in self._emails.values() if email.is_native]

    @property
    def external(self):
        return [email for email in self._emails.values() if email.is_external]

    @property
    def confirmed_external(self):
        return [email for email in self._emails.values()
                if email.is_confirmed and email.is_external]

    @property
    def suitable_for_restore(self):
        return [email for email in self._emails.values() if email.is_suitable_for_restore]


def build_emails(items):
    emails = Emails()

    for item in items:
        email = Email(
            is_native=item['native'],
            is_confirmed=item['validated'],
            is_rpop=item['rpop'],
            is_unsafe=item['unsafe'],
            is_default=item['default'],
            is_silent=item['silent'],
            address=item['address'],
        )
        emails.add(email)

    return emails


def domain_from_email(email):
    if '@' not in email:
        return ''
    domain = email.split('@', 1)[1]
    return domain


def punycode_email(email):
    """
    Закодируем доменную часть в punycode.
    Если доменной части нет, или она невалидная, ничего не делаем.
    """
    encoded_email = email
    if '@' in email:
        local, domain = email.rsplit('@', 1)
        try:
            domain = domain.encode('idna').decode('utf-8')
            encoded_email = u'%s@%s' % (local, domain)
        except UnicodeError:
            # невалидный имейл, возвращаем что было
            pass
    return encoded_email


def unicode_email(email):
    """
    Закодируем доменную часть в unicode.
    Если доменной части нет, или она невалидная, ничего не делаем.
    """
    decoded_email = email = smart_unicode(email)
    if '@' in email:
        local, domain = email.rsplit('@', 1)
        try:
            domain = domain.encode('utf-8').decode('idna')
            decoded_email = u'%s@%s' % (local, domain)
        except UnicodeError:
            # невалидный имейл, возвращаем что было
            pass
    return decoded_email


def normalize_email(email):
    """
    При проверке качества пароля, договорились нормализовать email таким образом:
    * Разбиваем на логин и домен, если есть '@'
    * Нормализуем логин
    * Приводим в нижний регистр доменную часть
    * Собираем email из нормализованного логина и домена
    """
    domain = domain_from_email(email)
    if domain:
        login = raw_login_from_email(email)
        login = normalize_login(login)
        normalized = '%s@%s' % (login, domain.lower())
        return normalized

    return email


def is_yandex_email(email):
    """
    Проверяет, является ли данный email-адрес адресом на Яндексе.
    Адрес может быть закодирован в IDNA. Работает с валидными email-адресами.
    """
    email = punycode_email(email)
    domain = domain_from_email(email)
    if domain:
        encoded_native_domains = []
        for native_domain in settings.NATIVE_EMAIL_DOMAINS:
            encoded_native_domains.append(native_domain.encode('idna').decode('utf-8'))
        return domain.lower() in encoded_native_domains


def normalize_email_with_phonenumber(email, country=None):
    """
    Привести email, представленный в виде телефонного алиаса, в формат, поддерживаемый email-валидатором
    (например, 79121234567@yandex.ru). В случае, если email не представлен в виде телефонного алиаса,
    нормализуем его, как в функции normalize_email.
    """
    domain = domain_from_email(email)
    if domain:
        login = raw_login_from_email(email)
        # Если в логине есть буквы, не считаем его телефоном
        if not any(char.isalpha() for char in login):
            phone = parse_phone_number(login, country)
            if phone:
                normalized = '%s@%s' % (phone.digital, domain.lower())
                return normalized

        login = normalize_login(login)
        normalized = '%s@%s' % (login, domain.lower())
        return normalized
    return email


def mask_email_for_statbox(email):
    """
    Маскирование email-а пользователя для записи в Статбокс. Маскируется таким образом, чтобы нельзя было
    догадаться или легко подобрать значение. Правила маскирования:
     - доменная часть не маскируется;
     - в логинной части показываем не более maximum_showed_symbols первых символов;
     - минимальная длина замаскированной части логина - minimal_masked_length символов.
    """
    minimal_masked_length = 5
    maximum_showed_symbols = 5
    login = raw_login_from_email(email)
    if len(login) <= minimal_masked_length:
        # Для короткого email-а не показываем его настоящую длину
        login = '*' * minimal_masked_length
    else:
        login = login[:-minimal_masked_length]
        symbols_to_show = min(len(login), maximum_showed_symbols)
        login = login[:symbols_to_show] + '*' * (minimal_masked_length + len(login) - symbols_to_show)
    domain = domain_from_email(email)
    if domain:
        return '%s@%s' % (login, domain)
    return login


def mask_email_for_challenge(email, mask_domain=True):
    login = raw_login_from_email(email)
    domain = domain_from_email(email)
    if len(login) < 3:
        masked_login = '***'
    else:
        masked_login = '%s***%s' % (login[0], login[-1])

    if mask_domain:
        if '.' not in domain:
            masked_domain = '***'
        else:
            subdomain, domain_zone = domain.rsplit('.', 1)
            if len(subdomain) < 2:
                masked_domain = '***.%s' % domain_zone
            else:
                masked_domain = '%s***.%s' % (subdomain[0], domain_zone)
    else:
        masked_domain = domain
    return '@'.join([masked_login, masked_domain])


def get_default_native_emails(login):
    return set(['%s@%s' % (login, domain) for domain in settings.NATIVE_EMAIL_DOMAINS])
