# -*- coding: utf-8 -*-

from netaddr import IPAddress
import six


ZERO_IP = IPAddress('0.0.0.0')


class IP(object):
    def __init__(self, ip):
        if hasattr(ip, '_ip'):
            self._ip = ip._ip
        else:
            if six.PY34 and isinstance(ip, bytes):
                ip = ip.decode('utf-8')
            self._ip = ip and IPAddress(ip)

    @property
    def version(self):
        return self._ip.version

    @property
    def normalized(self):
        if not self._ip and self._ip != ZERO_IP:
            return self._ip
        if self._ip.is_ipv4_mapped():
            return str(self._ip.ipv4())
        else:
            return str(self._ip)

    @property
    def truncated(self):
        if not self._ip and self._ip != ZERO_IP:
            return self._ip
        if self._ip.version == 4 or self._ip.is_ipv4_mapped():
            return str(self._ip.ipv4())
        else:
            return '0'

    @property
    def is_loopback(self):
        if not self._ip:
            return False
        return self._ip.is_loopback()

    def as_url(self):
        if self.version == 4:
            return str(self)
        elif self.version == 6:
            return '[' + str(self) + ']'
        else:
            raise NotImplementedError()

    def __str__(self):
        return str(self.normalized)

    def __repr__(self):
        return '<IP %s>' % self._ip

    def __eq__(self, other):
        if isinstance(other, IPAddress):
            return self._ip == other
        else:
            return str(self._ip) == other

    def __ne__(self, other):
        return not self.__eq__(other)

    def __bool__(self):
        return bool(self._ip)

    __nonzero__ = __bool__
