# -*- coding: utf-8 -*-

import re

import mock
import passport.backend.core.types.login.login
from passport.backend.utils.importlib import import_object


class FakeLoginGenerator(object):
    """
    Примеры использования

        set_response_value('social', 'soc-login1')
        set_side_effect_value('phonish', ['phne-login1', 'phne-login2'])
    """

    generate_login_re = re.compile('generate_(.*)_login$')

    def __init__(self):
        self._mocks = dict()
        self._patches = list()

        for login_type, login_generator, import_path in self.get_login_generators_extra():
            type_mock_name = import_path.split('.')[-1]
            type_mock = mock.Mock(type_mock_name, wraps=login_generator)

            self._mocks[login_type] = type_mock

            patch = mock.patch(import_path, type_mock)
            self._patches.append(patch)

    @classmethod
    def get_login_generators(cls):
        """
        Выходные параметры

            Словарь, в котором значения -- функции для генерации логинов,
            а ключи -- типы аккаунтов

            dict(
                phonish=generate_phonish_login,
                ...
            )
        """
        return {t: g for t, g, _ in cls.get_login_generators_extra()}

    @classmethod
    def get_login_generators_extra(cls):
        login_types = list()
        for attr_name in dir(passport.backend.core.types.login.login):
            match = cls.generate_login_re.match(attr_name)
            if match:
                login_types.append(match.group(1))

        login_generators = list()
        for login_type in login_types:
            path = 'passport.backend.core.types.login.login.generate_%s_login' % login_type
            login_generators.append((login_type, import_object(path), path))
        return login_generators

    def start(self):
        for patch in self._patches:
            patch.start()

    def stop(self):
        for patch in reversed(self._patches):
            patch.stop()

    def set_response_value(self, login_type, value):
        type_mock = self._mocks[login_type]
        type_mock.return_value = value
        type_mock.side_effect = None

    def set_response_side_effect(self, login_type, side_effect):
        type_mock = self._mocks[login_type]
        type_mock.side_effect = side_effect
