# -*- coding: utf-8 -*-
"""
Способы нормализации логина зависят от типа аккаунта.
Социальные пользователи создаются с синтетическими логинами. Но для юзера такого логина не существует.
Префикс 'uid-' определяет, что логин синтетический.

"""
import base64
import math
import os
import re

from passport.backend.core.conf import settings
from passport.backend.utils.string import smart_text


RE_LOGIN_FROM_EMAIL = [
    (re.compile(r'\+.{0,}'), ''),
    (re.compile(r'\_'), '-'),
    (re.compile(r'\-{2,}'), '-'),
    (re.compile(r'\-+$'), ''),
    (re.compile(r'^\-+'), ''),
    (re.compile(r'^\d+'), ''),
    (re.compile(r'[^a-zA-Z0-9_\-\.]'), '.'),
    (re.compile(r'\.{2,}'), '.'),
    (re.compile(r'\.+$'), ''),
    (re.compile(r'^\.+'), ''),
    (re.compile(r'[\.\-]{2,}'), '.'),
]
TEST_YANDEX_LOGIN_PREFIXES = ('yandex-team', 'yndx')
SOCIAL_LOGIN_PREFIX = 'uid-'
PHONISH_LOGIN_PREFIX = 'phne-'
NEOPHONISH_LOGIN_PREFIX = 'nphne-'
YAMBOT_LOGIN_PREFIX = 'yambot-'
KOLONKISH_LOGIN_PREFIX = 'kolonkish-'
TEST_FRODO_LOGIN_PREFIXES = ('frodo-spam', 'frodo-pdd-spam', 'frodo-change-pass')
KIDDISH_LOGIN_PREFIX = 'kid-'


def normalize_login(login):
    if not login:
        return login

    login = login.strip().lower()
    if '@' in login:
        return login
    return login.replace('.', '-')


def raw_login_from_email(email):
    """Просто вынуть часть email до символа @"""
    return email.split('@', 1)[0] if '@' in email else email


def extract_clean_login_from_email(email):
    """
    Функция превращает username из email в яндексовый логин, удаляя/заменяя запрещённые символы и т.п.
    """
    if '@' not in email:
        return email
    username = raw_login_from_email(email)
    for pattern, repl in RE_LOGIN_FROM_EMAIL:
        username = re.sub(pattern, repl, username)
    return username


def login_is_scholar(login):
    normalized_login = smart_text(normalize_login(login))
    match = login_is_scholar.RE_SCHOLAR_LOGIN.match(normalized_login)
    return bool(match)

login_is_scholar.RE_SCHOLAR_LOGIN = re.compile(smart_text(r'^[а-яё][а-яё0-9]*$'), re.IGNORECASE | re.UNICODE)


def login_is_social(login):
    norm_prefix = SOCIAL_LOGIN_PREFIX.replace('-', '.')
    return login.startswith(SOCIAL_LOGIN_PREFIX) or login.startswith(norm_prefix)


def login_is_phonish(login):
    norm_prefix = PHONISH_LOGIN_PREFIX.replace('-', '.')
    return login.startswith(PHONISH_LOGIN_PREFIX) or login.startswith(norm_prefix)


def login_is_neophonish(login):
    norm_prefix = NEOPHONISH_LOGIN_PREFIX.replace('-', '.')
    return login.startswith(NEOPHONISH_LOGIN_PREFIX) or login.startswith(norm_prefix)


def login_is_yambot(login):
    norm_prefix = YAMBOT_LOGIN_PREFIX.replace('-', '.')
    return login.startswith(YAMBOT_LOGIN_PREFIX) or login.startswith(norm_prefix)


def login_is_kiddish(login):
    norm_prefix = KIDDISH_LOGIN_PREFIX.replace('-', '.')
    return login.startswith(KIDDISH_LOGIN_PREFIX) or login.startswith(norm_prefix)


def login_is_kolonkish(login):
    norm_prefix = KOLONKISH_LOGIN_PREFIX.replace('-', '.')
    return login.startswith(KOLONKISH_LOGIN_PREFIX) or login.startswith(norm_prefix)


def is_test_yandex_login(login):
    """
    Функция определяет является ли предъявленный логин тестовым
    """
    for prefix in TEST_YANDEX_LOGIN_PREFIXES:
        if normalize_login(login).startswith(prefix):
            return prefix


def _generate_synthetic_login(prefix, random_part_byte_length=5):
    synthetic_login = b'%s%s' % (
        prefix.encode('utf-8'),
        base64.b32encode(os.urandom(random_part_byte_length)).lower(),
    )
    return synthetic_login.decode('utf-8')


def generate_social_login():
    return _generate_synthetic_login(SOCIAL_LOGIN_PREFIX)


def generate_phonish_login():
    return _generate_synthetic_login(PHONISH_LOGIN_PREFIX)


def generate_neophonish_login():
    return _generate_synthetic_login(NEOPHONISH_LOGIN_PREFIX)


def generate_yambot_login():
    return _generate_synthetic_login(YAMBOT_LOGIN_PREFIX)


def generate_kolonkish_login():
    return _generate_synthetic_login(KOLONKISH_LOGIN_PREFIX)


def generate_kiddish_login():
    return _generate_synthetic_login(KIDDISH_LOGIN_PREFIX)


def _mask_inner_string(s):
    return '%s***%s' % (s[0], s[-1]) if s else s


def masked_login(login):
    if not login:
        return login
    if u'@' in login:
        parts = login.split(u'@', 1)
        main_part = parts[0]
        postfix = u'@' + parts[1]
    else:
        main_part = login
        postfix = u''
    if len(main_part) == 1:
        return login
    else:
        main_part = main_part[:int(math.ceil(len(main_part) / 3.0))]
    return u'{}***{}'.format(main_part, postfix)


def strongly_masked_login(login):
    """
    В отличие от masked_login, не раскрываем даже короткие логины.
    Но даём понять, если в логине есть доменная часть.
    """
    if '@' in login:
        login_part, domain_part = login.rsplit('@', 1)
        return '%s@%s' % (_mask_inner_string(login_part), _mask_inner_string(domain_part))
    return _mask_inner_string(login)


def is_test_frodo_login(login):
    """
    Функция определяет является ли предъявленный логин тестовым логином
    для взаимодействия с ФО.
    """
    for prefix in TEST_FRODO_LOGIN_PREFIXES:
        if normalize_login(login).startswith(prefix):
            return prefix


def extract_phonenumber_alias_candidate_from_login(login):
    """
    Извлекает из логина цифровой логин, который может быть цифровым алиасом аккаунта (а может и не быть).
    Выполняет проверки:
    - логин не содержит букв
    - доменная часть, если она есть, может быть использована с цифровым алиасом
    """
    domain = ''
    if '@' in login:
        login, _, domain = login.partition('@')

    is_digital_login = not any(char.isalpha() for char in login)
    if not is_digital_login:
        return
    try:
        domain = domain.encode('utf-8').decode('idna')
    except UnicodeError:
        pass

    if not domain or domain in settings.NATIVE_EMAIL_DOMAINS or domain in settings.ALT_DOMAINS:
        return ''.join([char for char in login if char.isdigit()])
