# -*- coding: utf-8 -*-

from collections import namedtuple

from passport.backend.core.conf import settings
from passport.backend.core.types.phone_number.phone_number import PhoneNumber
from passport.backend.utils.time import parse_datetime


_SimplePhone = namedtuple('Phone', 'id number is_confirmed is_secure validation_datetime pending_type pending_phone_id')


sorted_phones = lambda phones: sorted(phones, key=lambda phone: phone.id)


class SimplePhone(_SimplePhone):

    @property
    def is_suitable_for_restore(self):
        """
        До определенного момента не было понятия защищенного телефона, поэтому у части пользователей может
        быть установлено несколько подтвержденных телефонов, каждый из которых мы считаем защищенным.
        """
        return (
            self.is_confirmed and
            (self.is_secure or self.validation_datetime <= settings.PHONES_SECURE_SCHEME_ACTIVE_DATETIME)
        )


class SimplePhones(object):
    def __init__(self):
        self._by_number = dict()
        self._secure = None

    def add(self, phone):
        self._by_number[phone.number.e164] = phone

        if phone.is_secure:
            self._secure = phone

    @property
    def secure(self):
        return self._secure

    def find(self, number):
        return self._by_number.get(number.e164)

    def find_confirmed(self, number):
        result = self.find(number)

        if result and result.is_confirmed:
            return result

    @property
    def confirmed(self):
        return sorted_phones([phone for phone in self._by_number.values() if phone.is_confirmed])

    @property
    def suitable_for_restore(self):
        return sorted_phones([
            phone for phone in self._by_number.values() if phone.is_suitable_for_restore
        ])

    @property
    def is_empty(self):
        return not self._by_number


def build_phones(items, allow_impossible=False):
    phones = SimplePhones()

    for item in items:
        is_confirmed = item['valid'] == 'valid'
        phone = SimplePhone(
            id=int(item['id']),
            number=PhoneNumber.parse(item['number'], allow_impossible=allow_impossible),
            is_confirmed=is_confirmed,
            is_secure=is_confirmed and bool(int(item['secure'])),
            validation_datetime=parse_datetime(item['validation_date']),
            pending_type=item.get('pending_type'),
            pending_phone_id=item.get('pending_phoneid'),
        )
        phones.add(phone)

    return phones
