# -*- coding: utf-8 -*-

import abc
from itertools import chain
import re

from passport.backend.core.conf import settings
from passport.backend.core.exceptions import BaseCoreError
import phonenumbers
from phonenumbers import (
    format_by_pattern,
    format_number,
    is_possible_number,
    is_valid_number,
    length_of_national_destination_code,
    national_significant_number,
    number_type,
    PhoneNumberFormat,
)
from phonenumbers.phonenumberutil import (
    NumberParseException,
    region_code_for_number,
)
from six import (
    add_metaclass,
    string_types,
)


PHONE_LIKE_REGEX = re.compile(r'^[-+()0-9\s]+$')
KNOWN_LOCAL_PHONES_PREFIXES = ('380', '90', '375', '7')

# Маппинг числовых типов телефонов в человеко-читаемые строки.
# Весь список возможных типов смотри в классе phonenumberutil.PhoneNumberType.
PHONE_NUMBER_TYPE_STRINGS = {
    0: 'fixed_line',
    1: 'mobile',
    2: 'fixed_line_or_mobile',
    3: 'toll_free',
    4: 'premium_rate',
    5: 'shared_cost',
    6: 'voip',
    7: 'personal_number',
    8: 'pager',
    9: 'uan',
    10: 'voicemail',
    99: 'unknown',
}


def initialize():
    """
    Подгрузить динамически либы phonenumbers,

    В чём проблема:
    phonenumbers подгружает свои либы динамически:

    $ virtualenv .venv
    $ source .venv/bin/activate
    (.venv)$ pip install -U -I phonenumbers==7.1.0
    (.venv)$ rm .venv/lib/python2.7/site-packages/phonenumbers/data/region_UA.{py,pyc}
    (.venv)$ ipython
    In [1]: import phonenumbers
    In [2]: phonenumbers.parse('+3807379649')  # UA номер
    --------------------------------------------------------------------------
    ImportError                              Traceback (most recent call last)
    ...
         21 def _load_region(code):
         22     __import__("region_%s" % code, globals(), locals(),
    ---> 23                fromlist=["PHONE_METADATA_%s" % code], level=1)
         24
         25 for region_code in _AVAILABLE_REGION_CODES:

    ImportError: No module named region_UA

    Te падает на вызове
    In [2]: phonenumbers.parse('+3807379649')  # UA номер
    а не на
    In [1]: import phonenumbers
    Такое поведение не всегда желательно
    Вызывая initialize мы подргужаем все либы
    region_UA.py. region_RU и тд ...
    """
    phonenumbers.PhoneMetadata.load_all()


def mask_for_statbox(raw_number):
    unmasked_count = 6  # ещё '+'
    if not raw_number or len(raw_number) < unmasked_count:
        return raw_number
    return "%s%s" % (raw_number[:unmasked_count], '*' * (len(raw_number) - unmasked_count))


def mask_phone_number(raw_number):
    """
    Основной метод маскировки номера. Маскируются цифры с третьей по седьмую с конца номера.
    """
    current, start_mask, end_mask = 0, 3, 7
    result = []
    reversed_number = raw_number[::-1]
    for symbol in reversed_number:
        if symbol.isdigit():
            current += 1
        if symbol.isdigit() and start_mask <= current <= end_mask:
            result.append('*')
        else:
            result.append(symbol)
    return ''.join(result[::-1])


def mask_for_flash_call(raw_number):
    """Маскируются последние 4 цифры номера"""
    total_digits = sum(1 for c in raw_number if c.isdigit())
    seen_digits = 0
    new_number = []
    for c in raw_number:
        if c.isdigit():
            seen_digits += 1
        if c.isdigit() and seen_digits > total_digits - 4:
            new_number.append('X')
        else:
            new_number.append(c)
    return ''.join(new_number)


def get_alt_phone_numbers_of_phone_number(phone_number):
    """
    Строит список других номеров телефонов, которые фактически звонят одному
    абоненту.

    Например, когда в стране меняют телефонную нумерацию, то для
    каждого старого номера появляется новый, такой что позвонив на старый или
    новый номер попадаешь к одному абоненту.
    """
    alts = list()

    for rule in PhoneNumbering.RULES:
        alt = None
        if rule.e164_is_new(phone_number.e164):
            alt = rule.new_e164_to_old(phone_number.e164)
        elif rule.e164_is_old(phone_number.e164):
            alt = rule.old_e164_to_new(phone_number.e164)

        if alt is not None:
            try:
                alts.append(PhoneNumber.parse(alt))
            except InvalidPhoneNumber:
                pass

    return alts


class InvalidPhoneNumber(BaseCoreError):
    pass


class PhoneNumber(object):
    # Оригинальное значение строки с номером, которое было распарсено при создании объекта.
    # Например, если при создании объекта был передан введённый пользователем номер "+ 7 911~22-33 44",
    # который был распарсен как номер "+79112223344", то здесь можно узнать то изначальное пользовательское значение.
    original = None

    # Оригинальное значение идентификатора страны, переданное при инициализации.
    original_country = None

    def __init__(self, raw_number, parsed_number, country):
        self.original = raw_number
        self.original_country = country.lower() if country else None
        self._parsed = parsed_number

    @classmethod
    def from_deprecated(cls, phone_number):
        """
        Создать соответствующий старому действующий новый номер

        Если номер уже новый, то он и будет возвращён
        """
        for rule in PhoneNumbering.RULES:
            if rule.is_old_deprecated() and rule.e164_is_old(phone_number.e164):
                try:
                    phone_number = PhoneNumber.parse(rule.old_e164_to_new(phone_number.e164))
                    break
                except InvalidPhoneNumber:
                    # Номер пока считается недействительным, скорее всего его
                    # забыли добавить в белый список.
                    pass
        return phone_number

    @staticmethod
    def is_like_yandex_countries_local_phone(raw_number):
        """
        Анализирует, похоже ли начало номера, указанного без знака "+", на один из кодов стран присутствия Яндекса.
        """
        return (isinstance(raw_number, string_types) and
                raw_number.lstrip('( ').startswith(KNOWN_LOCAL_PHONES_PREFIXES))

    @staticmethod
    def in_whitelist(number, whitelist):
        """
        Проверяет, входит ли отпарсенный номер в список исключений невалидных номеров.
        """
        return any(regex.match(number.e164) for regex, _ in whitelist)

    @staticmethod
    def in_fakelist(number, fakelist):
        """
        Проверяет номер на тестовость.
        """
        if isinstance(number, PhoneNumber):
            number = number.e164
        return any(regex.match(number) for regex, _ in fakelist)

    @staticmethod
    def in_blacklist(number):
        blacklist = [r for r in settings.PHONE_NUMBERS_WHITELIST if r[1] in settings.PHONE_NUMBERS_WHITELIST_DISABLED_RULES]
        return any(regex.match(number.e164) for regex, _ in blacklist)

    # TODO Кажется, этому методу не место в самом классе телефона, см. ниже комментарии к parse_phone_number.
    @classmethod
    def parse(cls, raw_number, country=None, allow_impossible=False,
              invalid_whitelist=None, fakelist=None):
        """
        Парсит строку с номером телефона и создаёт заполненный объект, если номер удалось распарсить. Если
        оригинальную строчку распарсить не удалось, пробует применить нехитрую эвристику с помощью подстановки
        знака "+" для номеров, начинающихся на код страны из списка стран присутствия Яндекса.

        :param raw_number: Строка с номером телефона в любом формате.
        :param country: Возможный идентификатор страны, которому может принадлежать номер. Может быть использован,
        если номер, например, введён в национальном формате, без указания кода страны.
        :param allow_impossible: Флаг, которым можно отключить проверку на валидность номера. Тогда объект с номером
        будет создан в любом случае, но часть его методов может не работать или работать не совсем корректно.
        :param invalid_whitelist: Список правил с исключениями для невалидных номеров. По умолчанию, используется
        список из конфига, который можно переопределить этим параметром.
        :param fakelist: Список правил для тестовых номеров

        :raise InvalidPhoneNumber: Выбрасывается, если ни одна попытка распарсить номер не удалась.
        """
        invalid_whitelist = invalid_whitelist or settings.PHONE_NUMBERS_WHITELIST
        invalid_whitelist = [
            [r, n] for r, n in invalid_whitelist if n not in settings.PHONE_NUMBERS_WHITELIST_DISABLED_RULES
        ]

        possible_raw_numbers = [(raw_number, country)]
        if cls.is_like_yandex_countries_local_phone(raw_number):
            possible_raw_numbers.append(('+' + raw_number, None))

        for possible_raw_number, possible_country in possible_raw_numbers:
            if possible_country:
                possible_country = possible_country.upper()

            try:
                parsed_number = phonenumbers.parse(possible_raw_number, possible_country)
            except NumberParseException:
                continue

            result = cls(raw_number, parsed_number, possible_country)

            fakelist = fakelist or settings.PHONE_NUMBERS_FAKELIST
            if cls.in_fakelist(result, fakelist):
                result = FakePhoneNumber(raw_number, parsed_number, possible_country)

            is_passed = (
                allow_impossible or
                (result.is_valid and not cls.in_blacklist(result)) or
                cls.in_whitelist(result, invalid_whitelist) or
                isinstance(result, FakePhoneNumber)
            )

            if is_passed:
                return result

        raise InvalidPhoneNumber('Invalid phone number', raw_number)

    @property
    def is_possible(self):
        """
        Флаг возможности существования такого номера в стране. Такая проверка снисходительнее и мягче, чем is_valid,
        плюс быстрее выполняется. По сути, внутри логика примерно такая:
          1. Проверяется только общая длина номера. При этом, начальные цифры национального номера (всё, что после
             кода страны), никак не валидируются.
          2. Не происходит попытки выяснить тип номера, то есть используются только общие правила всех номеров страны.
        Если равен False, то значение is_valid тоже будет False.
        """
        return is_possible_number(self._parsed)

    @property
    def is_valid(self):
        """
        Флаг полной валидности номера с точки зрения libphonenumber. Более жёсткая проверка, чем is_possible,
        поэтому и работает дольше. Означает, что номер включён в зарегистрированную номерную ёмкость, тип которой
        известен, а также, может быть известен мобильный оператор и/или регион страны.
        Если равен True, то значение is_possible тоже будет True.
        """
        return is_valid_number(self._parsed)

    @property
    def number_type_id(self):
        """
        Тип номера числом. Возможные значения см. в phonenumber.PhoneNumberType.
        """
        return number_type(self._parsed)

    @property
    def number_type(self):
        """
        Тип номера в человеко-читаемом строковом виде.
        """
        return PHONE_NUMBER_TYPE_STRINGS.get(self.number_type_id, 'unexpected')

    @property
    def national_significant_number(self):
        """
        Строка с национальной частью номера, так называемый National Significant Number или N(S)N,
        то есть без префикса с кодом страны. Например, для номера "+79112223344" - это "9112223344".
        """
        return national_significant_number(self._parsed)

    @property
    def national(self):
        """
        Отформатированная строка с номером в национальном формате, то есть так, как его используют
        для вызова внутри страны, включая разбиение номера на блоки с помощью различных разделителей.
        Например, для номера "+79112223344" - это "8 (911) 222-33-44".
        """
        return format_number(self._parsed, PhoneNumberFormat.NATIONAL)

    @property
    def international(self):
        """
        Отформатированная строка с номером в международном формате, включая разбиение номера на блоки с помощью
        различных разделителей. Например, для номера "+79112223344" - это "+7 911 222-33-44".
        """
        return format_number(self._parsed, PhoneNumberFormat.INTERNATIONAL)

    @property
    def e164(self):
        """
        Строка с номером в стандартизированном ITU-T формате E.164. Похожа на международнй формат,
        но без разбиения номера на блоки. Например, для номера "+79112223344" - это "+79112223344".
        """
        return format_number(self._parsed, PhoneNumberFormat.E164)

    @property
    def digital(self):
        """
        Строка с номером, содержащая только его цифры, без префиксного "+" и остальных разделителей.
        Например, для номера "+79112223344" - это "79112223344".
        """
        return self.e164.lstrip('+')

    @property
    def country(self):
        """
        Двухбуквенный идентификатор страны (в верхнем регистре), определённой автоматически по самому
        телефонному номеру. Например, для номера "+79112223344" - это "RU".
        """
        return region_code_for_number(self._parsed)

    @property
    def country_code(self):
        """
        Строка с числовым кодом страны из номера телефона. Например, для номера "+79112223344" - это 7.
        """
        return str(self._parsed.country_code)

    @property
    def national_destination_code_length(self):
        """
        Длина национального кода пункта назначения, то есть National Destination Code (NDC).
        Например, для номера "+79112223344" - это 3, т.к. его NDC = "911".
        """
        return length_of_national_destination_code(self._parsed)

    @property
    def national_destination_code(self):
        """
        Строка с национальным кодом пункта назначения, то есть National Destination Code (NDC).
        Например, для номера "+79112223344" - это "911".
        """
        return self.national_significant_number[:self.national_destination_code_length]

    @property
    def masked_format_for_frodo(self):
        """
        Строка с номером, содержащая только цифры, в котором занулены три первые цифры номера
        абонента (Subscribe Number). То есть, в строке полностью виден точный код страны, код
        пункта назначения (если таковой присутствует) и хвостовая часть номера.
        Например, для номера "+79112223344" - это "79110003344".
        """

        # Разбор на примере номера "+79112223344":
        cc = self.country_code                           # Код страны = "7"

        nsn = self.national_significant_number           # Национальный номер = "9112223344"
        nsn_len = len(nsn)                               # Длина национального номера = 10

        ndc_len = self.national_destination_code_length  # Длина кода пункта назначения = 3 ("911")
        zeroes_len = min(3, nsn_len - ndc_len)           # Длина занулённой части = 3 ("222")

        ndc = self.national_destination_code             # Код пункта назначения = "911"
        zeroes = '0' * zeroes_len                        # Занулённая часть = "000" (вместо "222")
        tail = nsn[ndc_len + zeroes_len:]                # Оставшийся незанулённый хвост номера = "3344"

        result = ''.join((cc, ndc, zeroes, tail))        # Код страны + код пункта назначения + нули + хвост = "79110003344"

        return result

    @property
    def masked_format_for_statbox(self):
        """
        Строка с номером в формате E.164, в которой видны только первые пять цифр номера, а каждая
        последующая цифра заменена символом "*". Например, для номера "+79112223344" - это "+79112******".
        """
        return mask_for_statbox(self.e164)

    @property
    def masked_format_for_challenge(self):
        """
        Строка с номером в формате international, в которой видны первые 4 цифры и 2 последние цифры номера,
        остальные цифры заменены символами "*". Например, для номера "+79112223344" - это "+7 911 ***-**-44".
        """
        return mask_phone_number(self.international)

    @property
    def masked_for_flash_call(self):
        return mask_for_flash_call(self.international)

    def is_similar_to(self, raw_number, country):
        """
        Проверяет, будет ли переданный чистый номер телефона распарсен так, что в результате
        получится объект с тем же номером, что лежит в нашем объекте.
        :param raw_number: Чистое, нераспарсенное значение номера телефона.
        :param country: Возможное значение страны для подсказки парсеру.
        """
        if not PHONE_LIKE_REGEX.match(raw_number):
            return False

        countries = [country, self.country]
        numbers = [parse_phone_number(raw_number, c)
                   for c in countries]

        return self in numbers

    def as_dict(self, only_masked=False):
        dump = dict(
            masked_original=mask_phone_number(self.original),
            masked_international=mask_phone_number(self.international),
            masked_e164=mask_phone_number(self.e164),
        )
        if not only_masked:
            dump.update(
                dict(
                    original=self.original,
                    international=self.international,
                    e164=self.e164,
                ),
            )
        return dump

    def __str__(self):
        return self.international

    def __repr__(self):
        return '<PhoneNumber: %s>' % str(self)

    def __eq__(self, other):
        if not isinstance(other, PhoneNumber):
            return False
        return self.e164 == other.e164

    def __ne__(self, other):
        return not self.__eq__(other)

    def __hash__(self):
        return hash(self.e164)

    def __int__(self):
        return int(self.digital)


class FakePhoneNumber(PhoneNumber):
    @property
    def national(self):
        return format_by_pattern(
            self._parsed,
            PhoneNumberFormat.NATIONAL,
            settings.NATIONAL_FAKE_PHONE_NUMBER_FORMATS,
        )

    @property
    def international(self):
        return format_by_pattern(
            self._parsed,
            PhoneNumberFormat.INTERNATIONAL,
            settings.INTERNATIONAL_FAKE_PHONE_NUMBER_FORMATS,
        )


@add_metaclass(abc.ABCMeta)
class PhoneTranslationRule(object):
    """
    Правило преобразования старых номеров в новые и обратно
    """

    @abc.abstractmethod
    def e164_is_new(self, e164):
        """
        Признак, что данный номер описывается правилом и считается новым

        Постусловия:

        e164_is_new(N) == e164_is_old(new_e164_to_old(N))
        """

    @abc.abstractmethod
    def e164_is_old(self, e164):
        """
        Признак, что данный номер описывается правилом и считается старым

        Постусловия:

        e164_is_old(N) == e164_is_new(old_e164_to_new(N))
        """

    @abc.abstractmethod
    def is_old_deprecated(self):
        """
        Признак, что описанные правилом старые номера следует избегать, т.е. по
        возможности не заводить новых старых номеров.

        Например, старые номера можно начать избегать, как только у
        пользователей появляется возможность использовать новые номера.
        """

    @abc.abstractmethod
    def new_e164_to_old(self, e164):
        """
        Преобразует данный новый номер в старый

        Программа не проверяет, что данный номер описывается правилом и
        считается новым. Чтобы проверить это используйте метод e164_is_new
        """

    @abc.abstractmethod
    def old_e164_to_new(self, e164):
        """
        Преобразует данный старый номер в новый

        Программа не проверяет, что данный номер описывается правилом и
        считается старым. Чтобы проверить это используйте метод e164_is_old
        """


class Brazil2016PhoneTranslationRule(PhoneTranslationRule):
    AREA_CODES = [
        range(11, 20),
        [21, 22, 24, 27, 28],
        range(31, 36),
        [37, 38],
        range(41, 50),
        range(51, 56),
        range(61, 70),
        [71, 73, 74, 75, 77, 79],
        range(81, 90),
        range(91, 100),
    ]
    AREA_CODES = [str(c) for c in chain(*AREA_CODES)]

    def e164_is_new(self, e164):
        return len(e164) == 14 and e164[1:3] == '55' and e164[3:5] in self.AREA_CODES and e164[5] == '9'

    def e164_is_old(self, e164):
        return self.e164_is_new(self.old_e164_to_new(e164))

    def is_old_deprecated(self):
        return True

    def new_e164_to_old(self, e164):
        return e164[:5] + e164[6:]

    def old_e164_to_new(self, e164):
        return e164[:5] + '9' + e164[5:]


class IvoryCoast2021PhoneTranslationRule(PhoneTranslationRule):
    def e164_is_new(self, e164):
        if len(e164) == 14:
            old_e164 = e164[:4] + e164[6:]
            return self.e164_is_old(old_e164) and e164 == self.old_e164_to_new(old_e164)
        else:
            return False

    def e164_is_old(self, e164):
        return len(e164) == 12 and e164[1:4] == '225' and e164[4] in '0456789'

    def is_old_deprecated(self):
        return settings.IS_IVORY_COAST_8_DIGIT_PHONES_DEPRECATED

    def new_e164_to_old(self, e164):
        return e164[:4] + e164[6:]

    def old_e164_to_new(self, e164):
        if e164[5] in '0123':
            prefix = '01'
        elif e164[5] in '456':
            prefix = '05'
        else:
            prefix = '07'
        return e164[:4] + prefix + e164[4:]


class PhoneNumbering(object):
    RULES = [
        Brazil2016PhoneTranslationRule(),
        IvoryCoast2021PhoneTranslationRule(),
    ]


# TODO Думаю, что parse_phone_number и PhoneNumber.parse должны быть объединены в какой-нибудь билдер/фабрику/конвертор.
# TODO Сейчас смотрится очень странно - одни и те же три слова, но в разном порядке, не говорящих ничего о том,
# TODO что именно парсится и что должно быть на выходе.
def parse_phone_number(number, country=None, allow_impossible=False):
    if isinstance(number, PhoneNumber):
        return number

    try:
        return PhoneNumber.parse(number, country=country, allow_impossible=allow_impossible)
    except InvalidPhoneNumber:
        return
