# -*- coding: utf-8 -*-
from base64 import (
    b32decode,
    b32encode,
    urlsafe_b64encode,
)
import os
import random

from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.utils.crc import (
    bytes_to_long,
    crc12,
    long_to_bytes,
)
from passport.backend.utils.string import (
    smart_bytes,
    smart_text,
)


APP_SECRET_LENGTH = 16  # 128 бит
PIN_DIGITS_MIN_COUNT = 4
PIN_DIGITS_MAX_COUNT = 16
MAX_DEFAULT_PIN = 10 ** PIN_DIGITS_MIN_COUNT - 1  # автосгенерённые пины должны быть короткими
MAX_PIN = 10 ** PIN_DIGITS_MAX_COUNT - 1


class InvalidPinError(BaseCoreError):
    pass


def parse_pin_as_string(pin):
    if not smart_bytes(pin).isdigit():
        raise InvalidPinError('Pin not an integer: %s' % pin)
    pin_value = int(pin)
    if pin_value < 0 or pin_value > MAX_PIN:
        raise InvalidPinError('Bad pin: %s' % pin)
    return smart_text(pin)


def get_zero_padded_pin(pin, length=None):
    pin_str = str(pin)
    length = length or PIN_DIGITS_MAX_COUNT
    if len(pin_str) > length:
        raise ValueError('Pin too long: %s' % pin)
    return '0' * (length - len(pin_str)) + pin_str


class TotpSecretType(object):
    @classmethod
    def generate(cls):
        return cls(
            app_secret=os.urandom(APP_SECRET_LENGTH),
            pin=get_zero_padded_pin(
                pin=int(random.SystemRandom().uniform(0, MAX_DEFAULT_PIN)),
                length=PIN_DIGITS_MIN_COUNT,
            ),
        )

    @classmethod
    def build(cls, b32_app_secret, pin):
        pin = parse_pin_as_string(pin)
        b32_app_secret = smart_bytes(b32_app_secret)
        if len(b32_app_secret) % 8 != 0:
            # восстановим паддинг, если требуется
            b32_app_secret += b'=' * (8 - len(b32_app_secret) % 8)
        app_secret = b32decode(b32_app_secret)

        if len(app_secret) > APP_SECRET_LENGTH:
            raise ValueError('Wrong secret bytes count: %s' % len(app_secret))
        app_secret = b'\0' * (APP_SECRET_LENGTH - len(app_secret)) + app_secret
        return cls(
            app_secret=app_secret,
            pin=pin,
        )

    def __init__(self, encrypted_pin_and_secret=None, pin=None, app_secret=None):
        if encrypted_pin_and_secret is not None:
            self.encrypted_pin_and_secret = encrypted_pin_and_secret
            self._pin = None
            self.app_secret = None
        elif pin is not None and app_secret is not None:
            self.encrypted_pin_and_secret = None
            self._pin = str(pin)
            self.app_secret = app_secret
        else:
            raise ValueError('Either encrypted_pin_and_secret or separate pin and secret must be specified')

    @property
    def pin(self):
        return self._pin

    @property
    def human_readable_app_secret(self):
        return b32encode(self.app_secret).decode().rstrip('=')

    @property
    def base64_secret(self):
        return urlsafe_b64encode(self.app_secret).decode().rstrip('=')

    def build_container_for_yakey(self, uid):
        message_value = bytes_to_long(self.app_secret)  # 16 байт на секрет

        message_value <<= 8 * 8  # 8 байт на uid
        message_value += uid

        message_value <<= 4  # 4 бита на длину секрета
        message_value += len(self._pin) - 1  # вычитаем 1, так как пинов длины 0 нет, а 16 - есть.

        signature = crc12(message_value)
        message_value <<= 12  # и 12 бит на подпись, to rule them all
        message_value += signature

        return b32encode(long_to_bytes(message_value)).decode().rstrip('=')

    def __eq__(self, other):
        if not isinstance(other, TotpSecretType):
            return False
        return (
            self.encrypted_pin_and_secret == other.encrypted_pin_and_secret and
            self.pin == other.pin and
            self.app_secret == other.app_secret
        )

    def __ne__(self, other):
        return not self.__eq__(other)
