# -*- coding: utf-8 -*-
"""Модули для тестирования программ использующих UserAgent"""

import mock
from passport.backend.core.test.test_utils import single_entrant_patch
from passport.backend.core.useragent.sync import DNSError
import requests


@single_entrant_patch
class UserAgentFaker(object):
    """Изолирует UserAgent"""

    DEFAULT_DNS_QUERY_RESPONSE = ['127.0.0.1']

    def __init__(self):
        self._patches = []

        self._session_mock = mock.Mock(name='session')
        session = mock.Mock(return_value=self._session_mock)
        patch = mock.patch('passport.backend.core.useragent.sync.requests.Session', session)
        self._patches.append(patch)

        self._dns_mock = mock.Mock(name='dns')
        dns_resolver = mock.Mock(return_value=self._dns_mock)
        patch = mock.patch('passport.backend.core.useragent.sync.DNSResolver', dns_resolver)
        self._patches.append(patch)

    def set_responses(self, responses):
        """
        Задать ответы от внешних ресурсов, которые получит UserAgent

        responses -- список ответов от внешних ресурсов, которые получит
                     UserAgent.

        Например, чтобы задать окружение в котором на 1-ом запросе UserAgent
        столкнётся с проблемой разыменования доменного имени, на 2-ом запросе
        не дождётся ответа от веб-службы, а на третьем Http-ответ с кодом 200
        и текстом "hello johnny", можно вызвать этот метод так:

        f = UserAgentFaker()
        with f.fake(
            responses=[
                f.DnsError(),
                f.TimeoutError(),
                f.HttpResponse(status_code=200, content="hello johnny"),
            ]
        ):
            run_user_agent()
        """
        self._set_dns_responses(responses)
        self._set_session_responses(responses)

    def _set_session_responses(self, responses):
        effects = []
        for resp in responses:
            if self._requests_is_responsible_for_response(resp):
                effects.append(resp)
        self._session_mock.request.side_effect = effects

    def _requests_is_responsible_for_response(self, response):
        for class_ in [
            FakedConnectionError,
            FakedTimeoutError,
            FakedHttpResponse,
        ]:
            if isinstance(response, class_):
                ret = True
                break
        else:
            ret = False
        return ret

    def _set_dns_responses(self, responses):
        effects = []
        for resp in responses:
            if self._dns_is_responsible_for_response(resp):
                effects.append(resp)
            else:
                effects.append(self.DEFAULT_DNS_QUERY_RESPONSE)
        self._dns_mock.query.side_effect = effects

    def _dns_is_responsible_for_response(self, response):
        return isinstance(response, FakedDnsError)

    def start(self):
        for patch in self._patches:
            patch.start()

    def stop(self):
        for patch in reversed(self._patches):
            patch.stop()


class FakedHttpResponse(object):
    """UA получит от веб-службы такой Http-ответ"""
    def __init__(self, status_code, content=None):
        self.status_code = status_code
        self.content = content or ''


class FakedTimeoutError(requests.Timeout):
    """UA не дождётся ответа от веб-службы"""


class FakedConnectionError(requests.ConnectionError):
    """UA не сможет подключиться к веб-службе"""


class FakedDnsError(DNSError):
    """UA не сможет разыменовать доменное имя"""
