# -*- coding: utf-8 -*-

import errno
import logging
import socket
import time

from netaddr import IPAddress
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import BaseCoreError
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)
from passport.backend.core.logging_utils.loggers import (
    FAILED_RESPONSE_CODE,
    SUCCESS_RESPONSE_CODE,
    TIMEOUT_RESPONSE_CODE,
)
from passport.backend.core.useragent._urllib3 import PoolManager
from passport.backend.core.useragent.name import (
    DNSError,
    DNSResolver,
)
from passport.backend.utils.string import smart_str
import requests
from requests.adapters import HTTPAdapter
import six
from six import iteritems
from six.moves import (
    http_client as httplib,
    urllib_parse as urlparse,
    xrange,
)
from urllib3 import exceptions as urllib3_exceptions


log = logging.getLogger('passport.useragent')


class RequestError(BaseCoreError):
    pass


class CustomRequestsAdapter(HTTPAdapter):
    """
    Адаптер, позволяющий передать внутрь requests доменное имя для проверки сертификата
    """
    def init_poolmanager(self, connections, maxsize, block=False, **pool_kwargs):
        pool_kwargs.update(
            block=block,
            maxsize=maxsize,
            num_pools=connections,
        )
        self.poolmanager = PoolManager(**pool_kwargs)


@lazy_loadable(name='UserAgent')
class UserAgent(object):
    """
    Высокоуровневый http(s) агент, построенный поверх библиотеки requests

    Хранит в себе пул соединений по ранее сделанным запросам
    Используем его lazy_loadable версию для взаимодействия с внешними сервисами
    Для других использований, например, запроса различных url, необходимо инстанциировать его отдельно
    """
    request_error_class = RequestError

    def __init__(
        self,
        max_pool_size=None,
        dns_cache=None,
        connections_per_pool=None,
    ):
        self.max_pool_size = max_pool_size or settings.USERAGENT_MAX_POOL_SIZE
        self.dns = DNSResolver(cache=dns_cache)

        self.session = requests.Session()

        requests_adapter_kwargs = dict(
            # инициализуем HTTPAdapter, чтобы передавать в него максимальное количество пулов
            pool_connections=self.max_pool_size,
        )
        if connections_per_pool is not None:
            # Максимальное число соединений в одном пуле
            requests_adapter_kwargs.update(pool_maxsize=connections_per_pool)
        self.requests_adapter = CustomRequestsAdapter(**requests_adapter_kwargs)

        self.session.mount('http://', self.requests_adapter)
        self.session.mount('https://', self.requests_adapter)

    def request(self, method, url, timeout=10, retries=1, reconnect=False,
                graphite_logger=None, statbox_logger=None, max_redirects=None,
                **kwargs):
        if 'allow_redirects' not in kwargs:
            kwargs['allow_redirects'] = False

        if six.PY3 and isinstance(url, str):
            url = url.encode('utf-8')

        parsed_url = urlparse.urlsplit(url)

        log.debug('Preparing HTTP request to %s with timeout %f', parsed_url.hostname.decode('utf-8'), timeout)

        request_url = self._process_url(url, kwargs)
        for i in xrange(retries):
            try:
                return self._try_request(
                    method,
                    request_url,
                    kwargs,
                    timeout,
                    graphite_logger=graphite_logger,
                    statbox_logger=statbox_logger,
                    reconnect=reconnect,
                    max_redirects=max_redirects,
                )
            except self.request_error_class:
                if i >= retries - 1:
                    raise

    def get(self, *args, **kwargs):
        return self.request('get', *args, **kwargs)

    def post(self, *args, **kwargs):
        return self.request('post', *args, **kwargs)

    def head(self, *args, **kwargs):
        return self.request('head', *args, **kwargs)

    def put(self, *args, **kwargs):
        return self.request('put', *args, **kwargs)

    def delete(self, *args, **kwargs):
        return self.request('delete', *args, **kwargs)

    def _try_request(self, method, request_url, request_kwargs, timeout,
                     graphite_logger=None, statbox_logger=None,
                     reconnect=False, max_redirects=None):
        try:
            ip_list = list(self.dns.query(request_url.hostname))
            log.debug('DNS query for %s returned %s', request_url.hostname.decode('utf-8'), list(ip_list))
        except DNSError:
            ip_list = []

        if not ip_list:
            raise self.request_error_class('DNS returned no IPs for %s' % request_url.hostname.decode('utf-8'))

        url = urlparse.urlunsplit(request_url)
        end_time = time.time() + timeout
        error_message = None
        self.session.max_redirects = max_redirects if max_redirects is not None else requests.sessions.DEFAULT_REDIRECT_LIMIT

        for ip in ip_list:
            ip_url = self._generate_url(request_url, ip)

            request_kwargs['timeout'] = end_time - time.time()
            if request_kwargs['timeout'] <= 0:
                raise self.request_error_class('Timeout is <= 0')

            log.debug('Requesting %s with timeout %f', ip_url.decode('utf-8'), request_kwargs['timeout'])

            start_time = time.time()
            response = None
            error = None

            for name, stream in iteritems(request_kwargs.get('files', {})):
                if isinstance(stream, tuple):
                    # (filename, fileobj)
                    stream = stream[1]
                stream.seek(0)

            try:
                self.requests_adapter.poolmanager.set_ip_for_host(request_url.hostname, ip)

                if reconnect:
                    # Принудительно инвалидируем соединение из пула, которое мы сейчас возьмем
                    # Хотим порвать keep-alive и попробовать заново установить соединение
                    # и не долбиться в тот же хост при ошибке (VAULT-424)
                    self.requests_adapter.poolmanager.invalidate_host_connection(
                        request_url.hostname,
                        port=request_url.port,
                        scheme=request_url.scheme,
                    )

                response = self.session.request(method, url, **request_kwargs)
                return response
            except (requests.Timeout, requests.ConnectionError, requests.exceptions.ChunkedEncodingError) as e:
                if isinstance(e, requests.exceptions.SSLError):
                    # Ошибка проверки серверного сертификата - хотим заметить это сразу
                    error_message = 'SSL problem while requesting %s' % request_url.hostname.decode('utf-8')
                    log.error(error_message, exc_info=e)
                elif (
                    isinstance(e, requests.ConnectionError) and
                    len(e.args) > 0 and
                    isinstance(e.args[0], urllib3_exceptions.ProtocolError) and
                    len(e.args[0].args) > 1 and
                    isinstance(e.args[0].args[1], httplib.BadStatusLine) and
                    e.args[0].args[1].line == "''"
                ):
                    # PASSP-5247: сервер закрыл соединение, хотим меньше логов в этом случае
                    error_message = 'Socket was closed while requesting %s' % ip_url.decode('utf-8')
                    log.debug(error_message)
                elif (
                    isinstance(e, requests.ConnectionError) and
                    len(e.args) > 0 and
                    isinstance(e.args[0], urllib3_exceptions.ProtocolError) and
                    len(e.args[0].args) > 1 and
                    isinstance(e.args[0].args[1], socket.error) and
                    e.args[0].args[1].errno == errno.EACCES
                ):
                    # Может возникнуть, если нет дырок в HBF - Host Based Firewall
                    error_message = 'Permission denied while requesting %s' % ip_url.decode('utf-8')
                    log.error(error_message, exc_info=e)
                elif isinstance(e, requests.ConnectTimeout):
                    error_message = 'Connect timeout while requesting %s' % ip_url.decode('utf-8')
                    log.warning(error_message, exc_info=e)
                elif isinstance(e, requests.ReadTimeout):
                    error_message = 'Read timeout while requesting %s' % ip_url.decode('utf-8')
                    log.warning(error_message, exc_info=e)
                elif isinstance(e, requests.Timeout):
                    error_message = 'Timeout while requesting %s' % ip_url.decode('utf-8')
                    log.warning(error_message, exc_info=e)
                elif isinstance(e, requests.exceptions.ChunkedEncodingError):
                    error_message = 'Connection reset by peer while reading response from %s' % ip_url.decode('utf-8')
                    log.warning(error_message, exc_info=e)
                else:
                    error_message = 'Error requesting %s' % ip_url.decode('utf-8')
                    log.warning(error_message, exc_info=e)

                self.dns.invalidate(request_url.hostname)
                error = e
            except requests.RequestException:
                raise
            except socket.error as e:
                if e.errno == errno.ECONNRESET:
                    # PASSP-9890: сервер закрыл соединение, хотим меньше логов в этом случае
                    error_message = 'Socket was closed while reading response from %s' % ip_url.decode('utf-8')
                    log.debug(error_message)

                self.dns.invalidate(request_url.hostname)
                error = e
            finally:
                duration = time.time() - start_time
                network_error = bool(error)
                http_code = str(response.status_code) if response is not None else '0'
                self._write_to_statbox(statbox_logger, duration, http_code, network_error)
                self._write_to_graphite_log(
                    graphite_logger,
                    error,
                    duration,
                    http_code,
                    network_error,
                    request_url.hostname,
                    ip,
                )

        raise self.request_error_class(error_message)

    def _write_to_statbox(self, statbox_logger, duration, http_code, network_error):
        if statbox_logger is not None:
            statbox_logger.log(
                duration=duration,
                http_code=http_code,
                network_error=network_error,
            )

    def _write_to_graphite_log(self, graphite_logger, error, duration, http_code,
                               network_error, srv_hostname, srv_ipaddress):
        if graphite_logger is not None:
            if isinstance(error, requests.Timeout):
                result = TIMEOUT_RESPONSE_CODE
            elif error is None:
                result = SUCCESS_RESPONSE_CODE
            else:
                result = FAILED_RESPONSE_CODE
            graphite_logger.log(
                http_code=http_code,
                duration=duration,
                response=result,
                network_error=network_error,
                srv_hostname=srv_hostname.decode('utf-8'),
                srv_ipaddress=srv_ipaddress,
            )

    def _process_url(self, url, request_kwargs):
        """
        Check if ``url`` contains IP address and set Host header if required.

        Modifies ``request_kwargs`` to add 'Host' to headers if not already present,
        encodes headers and cookies.
        Returns result of ``urlparse.urlsplit(url)``
        """
        if six.PY3 and isinstance(url, str):
            url = url.encode('utf-8')
        request_url = urlparse.urlsplit(url)

        headers = request_kwargs.get('headers', {})
        cookies = request_kwargs.get('cookies', {})

        if 'Host' not in headers:
            host_enc = request_url.hostname.decode('utf-8').encode('idna')
            if request_url.port:
                headers['Host'] = b'%s:%d' % (host_enc, request_url.port)
            else:
                headers['Host'] = host_enc

        headers = {smart_str(k): smart_str(v) for k, v in iteritems(headers)}
        cookies = {smart_str(k): smart_str(v) for k, v in iteritems(cookies)}
        request_kwargs['headers'] = headers
        request_kwargs['cookies'] = cookies
        return request_url

    def _generate_url(self, split_url, ip):
        if six.PY34 and isinstance(ip, bytes):
            ip = ip.decode('utf-8')
        if IPAddress(ip).version == 6:
            # http://www.ietf.org/rfc/rfc2732.txt
            netloc = '[%s]' % ip
        else:
            netloc = ip
        if split_url.port:
            netloc += ':%s' % split_url.port
        if split_url.username:
            username = split_url.username.decode('utf-8')
            if split_url.password:
                username += ':%s' % split_url.password.decode('utf-8')
            netloc = '%s@%s' % (username, netloc)
        netloc = netloc.encode('utf-8')

        return urlparse.urlunsplit(split_url._replace(netloc=netloc))


def get_useragent():
    return LazyLoader.get_instance('UserAgent')  # pragma: no cover
