# -*- coding: utf-8 -*-

from passport.backend.core.conf import settings
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.models.account import Account
from passport.backend.utils.common import chop


def get_many_accounts_by_uids(uids, blackbox_builder, userinfo_args=None):
    """
    Получает список учётных записей и множество uid'ов, для которых не существует
    учётной записи.

    Исключения
        BlackboxInvalidResponseError
        BlackboxTemporaryError
        BlackboxUnknownError
    """
    userinfo_list, unknown_uids = get_many_userinfo_by_uids(uids, blackbox_builder, userinfo_args)
    accounts, more_unknown_uids = get_many_accounts_by_userinfo_list(userinfo_list)
    unknown_uids.update(more_unknown_uids)
    return accounts, unknown_uids


def get_many_userinfo_by_uids(uids, blackbox_builder, userinfo_args=None):
    """
    Получает список учётных записей и множество uid'ов, для которых не существует
    учётной записи.

    Исключения
        BlackboxInvalidResponseError
        BlackboxTemporaryError
        BlackboxUnknownError
    """
    if not uids:
        return list(), set()

    user_info_list = list()
    for uids_chunk in chop(list(uids), settings.BLACKBOX_MAX_UIDS_PER_REQUEST):
        user_info_chunk = blackbox_builder.userinfo(
            **dict(userinfo_args or {}, uids=uids_chunk)
        )

        # Userinfo возвращает сведения о нескольких пользователях как список
        # словарей, а сведения об одном пользователе как словарь. Чтобы не
        # делать исключений для одного пользователя, поместим единственный
        # словарь в список.
        if isinstance(user_info_chunk, dict):
            user_info_chunk = [user_info_chunk]

        user_info_list += user_info_chunk

    known_user_info_list = list()
    unknown_uids = set()
    for user_info in user_info_list:
        if user_info.get('uid') is not None:
            known_user_info_list.append(user_info)
        else:
            unknown_uids.add(int(user_info[u'id']))

    return known_user_info_list, unknown_uids


def get_many_accounts_by_userinfo_list(userinfo_list):
    """
    Получает список учётных записей и множество uid'ов, для которых не существует
    учётной записи.
    """
    accounts = list()
    unknown_uids = set()
    for userinfo in userinfo_list:
        try:
            accounts.append(build_account(userinfo))
        except UnknownUid:
            unknown_uids.add(int(userinfo[u'id']))

    return accounts, unknown_uids


def build_account(data):
    """
    Получить аккаунт из ответа ЧЯ.

    data
        ответ Чёрного ящика.
    """
    return Account().parse(data)
