# -*- coding: utf-8 -*-

import hashlib
import logging

from passport.backend.core.builders.shakur.shakur import (
    BaseShakurError,
    get_shakur,
)
from passport.backend.core.conf import settings
from passport.backend.utils.string import smart_bytes


PASSWORD_SHA_PREFIX_LENGTH = 8

log = logging.getLogger('passport.shakur')


def is_login_in_shakur_whitelist(login):
    whitelist_login_masks = settings.SHAKUR_WHITELIST_LOGIN_MASKS
    return any(login.startswith(login_mask) for login_mask in whitelist_login_masks)


def check_password_is_pwned(password):
    try:
        if settings.IS_SHAKUR_CHECK_DISABLED:
            return

        password = smart_bytes(password)
        encrypted_password = hashlib.sha1(password).hexdigest()
        prefix = encrypted_password[:PASSWORD_SHA_PREFIX_LENGTH]
        shakur_info = get_shakur().check_password(sha_prefix=prefix)

        if shakur_info['found']:
            password_found = any(
                password['sha1'].lower() == encrypted_password for password in shakur_info['passwords']
            )

            if len(shakur_info['passwords']) == settings.SHAKUR_LIMIT:
                log.exception("Shakur 'checkpassword' for prefix %s exceed (or equal to) response limit" % prefix)

        else:
            password_found = False

        log.debug('Sha password with prefix %s was found in Shakur: %s' % (prefix, password_found))
        return password_found
    except BaseShakurError as e:
        log.warning('Error during Shakur request: %s' % e)
        return False
