# -*- coding: utf-8 -*-
import re

from formencode.validators import (
    _,
    FancyValidator,
    Invalid,
)
from passport.backend.core.conf import settings
from passport.backend.core.validators.utils import is_printable_ascii
from pyisemail import is_email
from pyisemail.diagnosis import (
    RFC5321Diagnosis,
    ValidDiagnosis,
)


class SimpleEmailValidator(FancyValidator):
    strip = True
    not_empty = True
    messages = {
        'invalid': _('Invalid email address'),
    }

    # - логинная часть непустая и не содержит пробельных символов
    # - доменная часть содержит точку, не начинается и не оканчивается точкой, не содержит пробелов
    EMAIL_RE = re.compile(r'^(?P<login>\S+)@(?P<domain>[^\s.]\S*\.\S*[^\s.])$', re.UNICODE)

    def _to_python(self, value, state):
        self.assert_string(value, state)
        match = self.EMAIL_RE.match(value)
        if not match:
            raise Invalid(
                self.message('invalid', state),
                value,
                state,
            )

        if '..' in match.group('domain'):
            raise Invalid(
                self.message('invalid', state),
                value,
                state,
            )

        return value

    def is_empty(self, value):
        # Пустые значения - только None и пустая строка
        return value is None or value == ''


class ComprehensiveEmailValidator(FancyValidator):
    """
    Этот валидатор проверяет адрес на соответствие различным RFC и общую
    корректность. При этом пропускает адреса с юникодными алфавитами.
    """
    strip = True
    not_empty = True
    messages = {
        'invalid': _('Invalid email address'),
        'native': _('Address within Yandex native domain'),
        'nonascii': _('Non-ASCII symbols found'),
    }

    def __init__(self, allow_native=True, max_domain_level=None, blacklisted_domains=None, *args, **kwargs):
        self.allow_native = allow_native
        self.max_domain_level = max_domain_level
        self.blacklisted_domains = blacklisted_domains

    def format_extended_error_message(self, result):
        citations = ', '.join([
            ref.citation
            for ref in result.references
        ])

        return '%s [%s]' % (
            result.message,
            citations or 'no references',
        )

    def _to_python(self, value, state):
        self.assert_string(value, state)

        if not settings.ALLOW_NONASCII_IN_EMAILS and not is_printable_ascii(value):
            raise Invalid(self.message('nonascii', state), value, state)

        result = is_email(
            value,
            check_dns=False,
            diagnose=True,
        )
        # Принимаем случаи как явно корректных адресов, так и технически
        # корректных для отсылки через SMTP.
        if not isinstance(result, (ValidDiagnosis, RFC5321Diagnosis)):
            raise Invalid(
                '%s: %s' % (
                    value,
                    self.format_extended_error_message(result),
                ),
                value,
                state,
            )

        domain = value.split('@')[-1]
        if '.' not in domain:
            # Технически адрес без домена верхнего уровня в конце является
            # корректным, но для наших целей не подходит.
            raise Invalid(self.message('invalid', state), value, state)

        if not self.allow_native:
            # Проверяем, что переданный email не принадлежит одному из доменов
            # Яндекса.
            if domain.lower() in settings.NATIVE_EMAIL_DOMAINS:
                raise Invalid(self.message('native', state), value, state)

        if self.max_domain_level is not None:
            domain_level = len(domain.split('.'))
            for tld in settings.COMPLEX_TLDS:
                if domain.endswith('.' + tld):
                    domain_level -= 1
                    break
            if domain_level > self.max_domain_level:
                raise Invalid(self.message('invalid', state), value, state)

        if self.blacklisted_domains:
            if domain.lower() in self.blacklisted_domains:
                raise Invalid(self.message('invalid', state), value, state)

        return value

    def is_empty(self, value):
        # Пустые значения - только None и пустая строка
        return value is None or value == ''


class LooseEmailValidator(SimpleEmailValidator):
    """Этот валидатор проверяет правильность email только если передано значение, похожее на email"""

    def _to_python(self, value, state):
        if '@' not in value:
            return value
        return super(LooseEmailValidator, self)._to_python(value, state)


class ExternalEmailValidator(SimpleEmailValidator):
    """Дополнительно проверяет, что переданный email не яндексовый"""

    def _to_python(self, value, state):
        value = super(ExternalEmailValidator, self)._to_python(value, state)
        domain = value.split('@')[-1]
        if domain in settings.NATIVE_EMAIL_DOMAINS:
            raise Invalid(self.message('invalid', state), value, state)
        return value
