# -*- coding: utf-8 -*-

from formencode.schema import format_compound_error
from passport.backend.core.types.file import File
from passport.backend.core.validators.validators import (
    _,
    FancyValidator,
    FormValidator,
    Invalid,
)
from six import iteritems


class FileUploadSchemaValidator(FormValidator):
    """
    Chained-валидатор для валидации файлов, переданных в форме. Файлы,
    полученные из формы, передаются посредством состояния, преобразованы в
    core-тип File. Ожидаемый набор входных файлов задается в конструкторе
    следующим образом:
      chained_validators = [
          FileUploadSchemaValidator(
              file1=FileValidator(not_empty=False),
              file2=FileValidator(),
          )
      ], где file1, file2 - ожидаемые поля формы, FileValidator - валидатор для
    обработки файлов, соответствующих данному полю (см., например,
    SingleFileUploadValidator).
    При выполнении валидации заданные в конструкторе поля ищутся в переданных
    файловых полях и в виде тупла объектов типа File передаются в валидатор,
    заданный в конструкторе. Тупл может содержать любое число объектов, так
    как это допускается форматом HTTP-форм.
    """
    def __init__(self, **kwargs):
        for k, v in iteritems(kwargs):
            if not isinstance(v, FancyValidator):
                raise ValueError('Expected FancyValidator instance as a value')
        self.file_schema = kwargs
        super(FileUploadSchemaValidator, self).__init__()

    def _to_python(self, value_dict, state):
        new_value_dict = dict(value_dict)
        req_files = getattr(state, 'files', {})
        errors = {}
        for field, validator in iteritems(self.file_schema):
            file_tuple = req_files.get(field, ())
            if field in new_value_dict:
                raise ValueError('File field already on form')
            try:
                file_cleaned = validator.to_python(file_tuple, state)
            except Invalid as e:
                errors[field] = e
                continue
            new_value_dict[field] = file_cleaned
        if errors:
            raise Invalid(format_compound_error(errors), value_dict, state, error_dict=errors)
        return new_value_dict


class SingleFileUploadValidator(FancyValidator):
    """
    Валидатор для проверки наличия файла, а также проверки
    соответствия имени файла заданному набору расширений.
    При проверке расширения регистр не учитывается.
    """
    messages = {
        'badExtension': _('File extension not supported'),
        'fileTooLarge': _('File is too large'),
    }

    def __init__(self, allowed_ext=None, max_size=None, **kwargs):
        allowed_ext = allowed_ext or []
        self.allowed_ext = tuple(['.' + e.lower() for e in allowed_ext])
        self.max_size = max_size
        super(SingleFileUploadValidator, self).__init__(**kwargs)

    def _extension_valid(self, filename):
        if self.allowed_ext:
            return filename.lower().endswith(self.allowed_ext)
        return True

    def _file_size_valid(self, file):
        return self.max_size is None or len(file) <= self.max_size

    def _to_python(self, value, state=None):
        try:
            file = value[0]
        except TypeError:
            raise ValueError('Expected non-empty tuple, got %s' % value)

        if not isinstance(file, File):
            raise ValueError('Expected File object, got %s' % file)

        if not self._extension_valid(file.filename):
            raise Invalid(self.message('badExtension', state), file, state)
        if not self._file_size_valid(file):
            raise Invalid(self.message('fileTooLarge', state), file, state)
        return file

    def is_empty(self, value):
        try:
            return len(value) == 0
        except TypeError:
            return super(SingleFileUploadValidator, self).is_empty(value)
