# -*- coding: utf-8 -*-

from collections import OrderedDict

from formencode import (
    FancyValidator,
    Invalid,
)
import jsonschema
from jsonschema.validators import extend as jsonschema_extend
from passport.backend.core.validators.utils import convert_formencode_error_code
from six import iteritems


VALIDATORS_ORDERED = (
    'type',  # сначала проверим тип
    'required',  # далее - наличие элементов в объекте
    'items',  # затем элементы списка
    'properties',  # и поля объекта
    # затем всё остальное
)

VALIDATORS_ORDER_KEY = dict([(field, i) for (i, field) in enumerate(VALIDATORS_ORDERED)])


def _schema_sort_key(item):
    validator, _ = item
    return VALIDATORS_ORDER_KEY.get(validator, len(VALIDATORS_ORDERED))


def _sort_schema(schema):
    if not isinstance(schema, dict):
        return schema
    sorted_schema = OrderedDict(sorted(schema.items(), key=_schema_sort_key))
    for validator, subschema in iteritems(sorted_schema):
        sorted_schema[validator] = _sort_schema(subschema)
    return sorted_schema


class JsonSchema(object):
    """
    Вспомогательный класс для обозначения JSON-схемы.

    Json-схему нужно сортировать, т.к. мы добавляем модифицирующие операции и порядок валидации
    становится важен. Пример: мы strip'аем элементы списка-строки и требуем их уникальности.
    """
    def __init__(self, schema):
        self.schema = _sort_schema(schema)


class ValidationErrorAdapter(jsonschema.ValidationError):
    """
    Класс ошибки валидации json-схемы расширен указанием кода ошибки и поля, вызвавшего ошибку.
    """
    def __init__(self, code, message, field=None):
        super(ValidationErrorAdapter, self).__init__(message)
        self.error_code = code
        self.error_field = field


def _properties_draft4_extended(validator, properties, instance, schema):
    """
    Валидатор properties для объектов в json-схеме. Расширен возможностью указания поля со значением,
    являющимся formencode-валидатором. При этом происходит модификация данных.
    """
    if not validator.is_type(instance, 'object'):
        return
    for property, subschema in iteritems(properties):
        if isinstance(subschema, FancyValidator):
            if property in instance:
                try:
                    instance[property] = subschema.to_python(instance[property])
                except Invalid as e:
                    code = convert_formencode_error_code(e.code)
                    yield ValidationErrorAdapter(code, e.unpack_errors(), field=property)
        elif property in instance:
            for error in validator.descend(
                instance[property],
                subschema,
                path=property,
                schema_path=property,
            ):
                yield error


def _required_draft4_with_code(validator, required, instance, schema):
    """
    Валидатор required для объектов в json-схеме. Расширен указанием кода ошибки и имени поля.
    """
    if not validator.is_type(instance, 'object'):
        return
    for property in required:
        if property not in instance:
            yield ValidationErrorAdapter('empty', '%r is a required property' % property, field=property)


def _unique_items_with_code(validator, unique_items, instance, schema):
    """
    Валидатор, проверяющий уникальность элементов в списке. В коде ошибки указывает номер дублирующегося элемента.
    """
    if unique_items and validator.is_type(instance, 'array') and instance:
        for i, elem in enumerate(instance):
            if elem in instance[:i]:
                yield ValidationErrorAdapter(
                    'duplicate',
                    '%r has non-unique element at position %d' % (instance, i),
                )


def _items_extended(validator, items, instance, schema):
    """
    Валидатор items для элементов списка в json-схеме. Расширен возможностью задания formencode-валидатора
    в качестве значения items. В этом случае все элементы должны удовлетворять валидатору. Элементы
    могут модифицироваться.
    """
    if not validator.is_type(instance, 'array'):
        return

    if validator.is_type(items, 'object') or isinstance(items, FancyValidator):
        for index, item in enumerate(instance):
            if isinstance(items, FancyValidator):
                try:
                    instance[index] = items.to_python(item)
                except Invalid as e:
                    code = convert_formencode_error_code(e.code)
                    yield ValidationErrorAdapter(code, e.unpack_errors())
            else:
                for error in validator.descend(item, items):
                    yield error
    else:
        for (index, item), subschema in zip(enumerate(instance), items):
            for error in validator.descend(
                item, subschema, schema_path=index,
            ):
                yield error


# Расширенный валидатор json-документа
_PassportJsonSchemaValidator = jsonschema_extend(
    jsonschema.Draft4Validator,
    dict(
        required=_required_draft4_with_code,
        properties=_properties_draft4_extended,
        items=_items_extended,
        uniqueItems=_unique_items_with_code,
    ),
)


def validate_jsonschema_form(schema, form_data):
    """
    Проверить, что заданный объект удовлетворяет требованиям json-схемы.
    @param schema: JSON-схема
    @param form_data: объект для валидации
    @return список всех ошибок валидации
    """
    if not isinstance(schema, JsonSchema):
        raise TypeError('Expected object of type JsonSchema, got %s instead' % type(schema))
    errors = set()
    for error in _PassportJsonSchemaValidator(schema.schema).iter_errors(form_data):
        path = list(map(str, error.path))
        if isinstance(error, ValidationErrorAdapter):
            if error.error_field is not None:
                path.append(str(error.error_field))
            code = error.error_code
        else:
            code = 'invalid'
        errors.add('.'.join(path + [code]))

    return sorted(list(errors))
