# -*- coding: utf-8 -*

from formencode import NoDefault
from passport.backend.core.conf import settings
from passport.backend.core.language_detect import get_language_detect
from passport.backend.core.validators.validators import (
    FancyValidator,
    Invalid,
)
from passport.backend.utils.string import smart_bytes


class PortalLanguage(FancyValidator):
    strip = True
    not_empty = True
    messages = dict(unsupported=u'Language "%(language)s" is not supported')

    # Используется, когда язык задан, но его нет в списке.
    default_language = NoDefault

    def _to_python(self, value, state):
        value = value.lower()
        if value not in self.get_languages():
            if self.default_language is NoDefault:
                raise Invalid(self.message('unsupported', state, language=value), value, state)
            value = self.default_language
        return value

    def get_languages(self):
        return settings.PORTAL_LANGUAGES


class DisplayLanguage(PortalLanguage):
    def get_languages(self):
        return settings.DISPLAY_LANGUAGES


class DetectedLanguage(DisplayLanguage):

    def try_detect_language(self, language):
        language_request = {
            'filter': smart_bytes(','.join(self.get_languages())),
            'default': smart_bytes(settings.ALL_SUPPORTED_LANGUAGES['default']),
            'pass-language': smart_bytes(language or ''),
        }
        return get_language_detect().find(language_request)[0][0].decode('utf8')

    def _to_python(self, value, state):
        value = value.lower()
        if value not in self.get_languages():
            value = self.try_detect_language(value)
        return value


class CaptchaDisplayLanguage(PortalLanguage):

    def _to_python(self, value, state):
        value = value.lower()
        if value not in self.get_languages():
            return ''
        return value

    def get_languages(self):
        return settings.DISPLAY_LANGUAGES


class ConfirmationEmailLanguage(PortalLanguage):

    def get_languages(self):
        return settings.EMAIL_VALIDATOR_EMAIL_LANGUAGES
