# -*- coding: utf-8 -*-
import logging

from passport.backend.core.builders import blackbox
from passport.backend.core.grants.grants_config import check_specialtest_yandex_login_grant
from passport.backend.core.portallib import is_yandex_ip
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_KIDDISH,
    ACCOUNT_TYPE_KOLONKISH,
    ACCOUNT_TYPE_LITE,
    ACCOUNT_TYPE_NEOPHONISH,
    ACCOUNT_TYPE_NORMAL,
    ACCOUNT_TYPE_PDD,
    ACCOUNT_TYPE_PHONISH,
    ACCOUNT_TYPE_SCHOLAR,
    ACCOUNT_TYPE_SOCIAL,
    ACCOUNT_TYPE_YAMBOT,
)
from passport.backend.core.types.login.login import (
    is_test_yandex_login,
    login_is_kiddish,
    login_is_kolonkish,
    login_is_neophonish,
    login_is_phonish,
    login_is_scholar,
    login_is_social,
    login_is_yambot,
    normalize_login,
)
from passport.backend.core.validators.validators import (
    _,
    FormValidator,
    Invalid,
)


LOGIN_TYPE_PUBLIC_ID = 'public_id'

log = logging.getLogger('passport.validators')


class Availability(FormValidator):
    """
    Проверка на занятость логина
    """
    messages = {
        'notAvailable': _('Login is not available'),
    }

    login_field = 'login'
    domain_field = None
    ignore_stoplist_field = 'ignore_stoplist'

    def __init__(self, login_type=ACCOUNT_TYPE_NORMAL, uid=None, **kwargs):
        self._login_type = login_type
        self._uid = str(uid)
        super(Availability, self).__init__(**kwargs)

    def _to_python(self, value_dict, state):
        login = value_dict.get(self.login_field)
        if login is None:
            return value_dict

        ignore_stoplist = value_dict.get(self.ignore_stoplist_field)
        is_pdd = False

        message = self.message('notAvailable', state)

        not_available_error = Invalid(
            message,
            value_dict,
            state,
            error_dict={
                self.login_field: Invalid(message, value_dict, state),
            },
        )

        # Проверяем дополнительно логины на соответствие типу потому,
        # что для социальщиков и фонишей у нас есть функции генерации логина.
        # Сгенерированный логин не отдается на проверку валидатору логинов.
        if self._login_type == ACCOUNT_TYPE_NORMAL:
            if (
                login_is_kiddish(login) or
                login_is_kolonkish(login) or
                login_is_neophonish(login) or
                login_is_phonish(login) or
                login_is_social(login) or
                login_is_yambot(login) or
                login_is_scholar(login)
            ):
                raise not_available_error

            ip = state.env.user_ip
            if not ignore_stoplist:
                test_prefix = is_test_yandex_login(login)
                if test_prefix:
                    if is_yandex_ip(str(ip)) or check_specialtest_yandex_login_grant(ip, test_prefix):
                        ignore_stoplist = True
                    else:
                        raise not_available_error
        elif self._login_type == ACCOUNT_TYPE_SOCIAL:
            if not login_is_social(login):
                raise not_available_error
            ignore_stoplist = True
        elif self._login_type == ACCOUNT_TYPE_PDD:
            is_pdd = True
            if self.domain_field:
                domain = value_dict.get(self.domain_field)
                login = '%s@%s' % (login, domain)
        elif self._login_type == ACCOUNT_TYPE_PHONISH:
            if not login_is_phonish(login):
                raise not_available_error
            ignore_stoplist = True
        elif self._login_type == ACCOUNT_TYPE_NEOPHONISH:
            if not login_is_neophonish(login):
                raise not_available_error
            ignore_stoplist = True
        elif self._login_type == ACCOUNT_TYPE_YAMBOT:
            if not login_is_yambot(login):
                raise not_available_error
            ignore_stoplist = True
        elif self._login_type == ACCOUNT_TYPE_KIDDISH:
            if not login_is_kiddish(login):
                raise not_available_error
            ignore_stoplist = True
        elif self._login_type == ACCOUNT_TYPE_KOLONKISH:
            if not login_is_kolonkish(login):
                raise not_available_error
            ignore_stoplist = True
        elif self._login_type == ACCOUNT_TYPE_LITE:
            ignore_stoplist = True
        elif self._login_type == ACCOUNT_TYPE_SCHOLAR:
            if not login_is_scholar(login):
                raise not_available_error
            ignore_stoplist = True
        elif self._login_type == LOGIN_TYPE_PUBLIC_ID:
            login = normalize_login(login)
        else:
            raise ValueError('Bad login_type value: `%s`' % self._login_type)

        bb = blackbox.Blackbox()
        response = bb.loginoccupation([login], ignore_stoplist, is_pdd)

        if response[login]['status'] != 'free':
            # Разрешено занять public_id совпадающий с собственным логином
            if self._login_type != LOGIN_TYPE_PUBLIC_ID or response[login]['status'] != 'occupied' or response[login].get('uid') != self._uid:
                log.debug('Blackbox returned status=%s for login=%s', response[login], login)
                raise not_available_error

        return value_dict


class PublicIdAvailability(Availability):
    login_field = 'public_id'

    def __init__(self, **kwargs):
        super(PublicIdAvailability, self).__init__(login_type=LOGIN_TYPE_PUBLIC_ID, **kwargs)
