# -*- coding: utf-8 -*-
import re

from passport.backend.core.conf import settings
from passport.backend.core.types.login.login import login_is_social
from passport.backend.core.validators.email import ComprehensiveEmailValidator
from passport.backend.core.validators.validators import (
    _,
    FancyValidator,
    Invalid,
)
from passport.backend.utils.string import smart_text


MAX_LENGTH = 30  # Ограничение на длину логина в yandex.ru
MAX_PDD_LENGTH = 40
MAX_LITE_LENGTH = 40
MAX_YANDEX_TEAM_LENGTH = 100  # Ограничение на длину логина в yandex-team.ru

_ALIAS = {'.': 'Dot', '-': 'Hyphen'}

# максимальная гранулярность ошибок
ERROR_MESSAGES = {
    # общие ошибки
    'tooLong': _('Login has more than %d symbols' % MAX_LENGTH),
    'startsWithDigit': _('Login starts with digit'),
    'startsWithDot': _('Login starts with `.`'),
    'startsWithHyphen': _('Login starts with `-`'),
    'endsWithDot': _('Login ends with `.`'),
    'endsWithHyphen': _('Login ends with `-`'),
    'doubledDot': _('Login has doubled `.`'),
    'doubledHyphen': _('Login has doubled `-`'),
    'prohibitedSymbols': _('Login has prohibited symbols'),
    'dotHyphen': _('Login has `.-`'),
    'hyphenDot': _('Login has `-.`'),

    # специфичные ошибки
    'notSocial': _('Login is not social'),
    'idna': _('IDNA lite logins are prohibited'),
}


def _build_exception_with_errors(validator, value, state, error_codes, field_name):
    """
    Строим исключение, содержащее информацию обо всех переданных ошибках
    :param validator: Экземпляр валидатора, который знает как строить сообщения о ошибках
    :param error_codes: Список кодов ошибок
    """
    error_list = [
        Invalid(validator.message(msg_key, state), value, state)
        for msg_key in error_codes
    ]
    message = validator.message(error_codes[0], state)
    error_dict = {field_name: Invalid(message, value, state, error_list=error_list)}
    return Invalid(message, value, state, error_dict=error_dict)


class BaseValidator(FancyValidator):
    """
    Абстрактный класс для проверки введенной строки на соответствие правилам составления логина
    Вызывает функцию проверки всех возможных ошибок и составляет подробное сообщение о допущенных ошибках
    """

    messages = ERROR_MESSAGES

    # ограничение по минимальной длине
    not_empty = True
    strip = True

    @property
    def _field_name(self):
        raise NotImplementedError()  # pragma: no cover

    def _validate(self, value):
        raise NotImplementedError()  # pragma: no cover

    def _to_python(self, value, state):
        error_codes = self._validate(value)
        if error_codes:
            raise _build_exception_with_errors(self, value, state, error_codes, self._field_name)

        return value


class LooseLoginValidator(BaseValidator):
    """
    Валидация логина с самыми базовыми проверками.

    Нужна, например, при авторизации, когда мы должны пропускать даже логины, которые сейчас невалидны (но могли быть
    валидными раньше)
    """
    max_length = 255
    _field_name = 'login'

    def _validate(self, login):
        errors = []

        # ограничение по длине
        if len(login) > self.max_length:
            errors.append('tooLong')

        return errors


class Login(BaseValidator):
    """Проверка портального логина -- с почтовым ящиком на домене yandex.ru"""

    max_length = MAX_LENGTH
    prohibited_symbols_re = re.compile(r'[^a-zA-Z0-9.-]')
    _field_name = 'login'

    def _validate(self, login):
        errors = list()

        # ограничение по длине
        if len(login) > self.max_length:
            errors.append('tooLong')

        # не может начинаться с цифры
        if login[0].isdigit():
            errors.append('startsWithDigit')

        # не может начинаться на дефис и точку
        # используемый код работает быстрее аналогичного с регулярками или методами startswith и endswith
        if login[0] in ('.', '-'):
            errors.append('startsWith%s' % _ALIAS[login[0]])

        # не может заканчиваться на дефис и точку
        if login[-1] in ('.', '-'):
            errors.append('endsWith%s' % _ALIAS[login[-1]])

        # точки разрешены, но не подряд
        if '..' in login:
            errors.append('doubledDot')

        # дефисы разрешены, но не подряд
        if '--' in login:
            errors.append('doubledHyphen')

        # запрещены точка-тире
        if '.-' in login:
            errors.append('dotHyphen')

        # запрещены тире-точка
        if '-.' in login:
            errors.append('hyphenDot')

        # только латинские символы, цифры, точка и тире
        if self.prohibited_symbols_re.search(login):
            errors.append('prohibitedSymbols')

        return errors


class PddLogin(Login):
    """
    Проверка логина пользователя ПДД. Условия для них мягче, чем для
    портальных логинов. Частично основано на коде админки ПДД, частично
    взято из Перла.
    """

    max_length = MAX_PDD_LENGTH
    prohibited_symbols_re = re.compile(r'[^a-zA-Z0-9._-]')

    messages = dict(
        ERROR_MESSAGES,
        tooLong=_('Login has more than %d symbols' % max_length),
    )

    def _validate(self, login):
        errors = []

        # ограничение по длине
        if len(login) > self.max_length:
            errors.append('tooLong')

        # только латинские символы, цифры, точка, тире и нижнее подчеркивание
        if self.prohibited_symbols_re.search(login):
            errors.append('prohibitedSymbols')

        return errors


class SocialLogin(Login):
    """Синтетический логин социального пользователя"""
    strip = False

    def _validate(self, login):
        if not login_is_social(login):
            return ['notSocial']

        return super(SocialLogin, self)._validate(login)


class LiteLogin(BaseValidator):
    """
    Валидатор email адресов с доменов партнеров -- без проверок к логиновой части
    """
    max_length = MAX_LITE_LENGTH
    prohibited_symbols_re = re.compile(r'[^@a-zA-Z0-9._-]')
    _field_name = 'login'

    messages = dict(
        ERROR_MESSAGES,
        tooLong=_('Login has more than %d symbols' % max_length),
    )

    def _validate(self, value):
        errors = list()

        # ограничение по длине
        if len(value) > self.max_length:
            errors.append('tooLong')

        # только латинские символы, цифры, точка, тире и нижнее подчеркивание
        if self.prohibited_symbols_re.search(value):
            errors.append('prohibitedSymbols')

        # idna-домены запрещены
        if 'xn--' in value:
            errors.append('idna')

        return errors

    def _to_python(self, value, state):
        value = super(LiteLogin, self)._to_python(value, state)

        email_validator = ComprehensiveEmailValidator(
            allow_native=False,
            max_domain_level=settings.LITE_LOGIN_MAX_DOMAIN_LEVEL,
            blacklisted_domains=settings.LITE_LOGIN_BLACKLISTED_DOMAINS,
        )
        value = email_validator.to_python(value, state)

        return value.lower()


class YandexTeamLogin(BaseValidator):
    """Для логинов в домене yandex-team применяются другие правила"""
    prohibited_symbols_re = re.compile(r'[^a-zA-Z0-9_-]')
    max_length = MAX_YANDEX_TEAM_LENGTH
    _field_name = 'login'

    messages = dict(
        ERROR_MESSAGES,
        tooLong=_('Login has more %d symbols' % max_length),
    )

    def _validate(self, login):
        errors = list()

        # ограничение по длине
        if len(login) > self.max_length:
            errors.append('tooLong')

        # не может начинаться на дефис
        if login[0] == '-':
            errors.append('startsWith%s' % _ALIAS[login[0]])

        # не может заканчиваться на дефис
        if login[-1] == '-':
            errors.append('endsWith%s' % _ALIAS[login[-1]])

        # только латинские символы, цифры подчеркивание и тире
        if self.prohibited_symbols_re.search(login):
            errors.append('prohibitedSymbols')

        return errors


class ScholarLogin(BaseValidator):
    max_length = MAX_LENGTH
    prohibited_symbols_re = re.compile(smart_text(r'[^а-яё0-9]'), re.IGNORECASE | re.UNICODE)
    _field_name = 'login'

    def _validate(self, login):
        login = smart_text(login)
        errors = list()

        # ограничение по длине
        if len(login) > self.max_length:
            errors.append('tooLong')

        # не может начинаться с цифры
        if login[0].isdigit():
            errors.append('startsWithDigit')

        # только кириллица и цифры
        if self.prohibited_symbols_re.search(login):
            errors.append('prohibitedSymbols')

        return errors
