# -*- coding: utf-8 -*-
import re

from passport.backend.core.conf import settings
from passport.backend.core.models.password import (
    check_password_equals_hash,
    check_password_equals_sha256_hash,
    check_password_history,
    PASSWORD_ENCODING_VERSION_MD5_CRYPT,
    PASSWORD_ENCODING_VERSION_MD5_CRYPT_ARGON,
    PASSWORD_ENCODING_VERSION_MD5_HEX,
    PASSWORD_ENCODING_VERSION_RAW_MD5_ARGON,
)
from passport.backend.core.password import get_password_qualifier
from passport.backend.core.password.policy import (
    get_policy,
    MAX_PASSWORD_LENGTH,
    PASSWORD_POLICY_NAMES,
)
from passport.backend.core.types.email.email import normalize_email
from passport.backend.core.types.login.login import (
    normalize_login,
    raw_login_from_email,
)
from passport.backend.core.types.phone_number.phone_number import parse_phone_number
from passport.backend.core.utils.shakur.shakur import (
    check_password_is_pwned,
    is_login_in_shakur_whitelist,
)
from passport.backend.core.validators.validators import (
    _,
    FancyValidator,
    FormValidator,
    Invalid,
    OneOf,
    String,
)


MD5_CRYPT_REGEX = re.compile(r'^\$1\$[a-zA-Z0-9./]{8}\$[a-zA-Z0-9./]{22}$')
MD5_RAW_REGEX = re.compile(r'^[a-zA-Z0-9]{32}$')


class PasswordPolicy(FancyValidator):
    def _to_python(self, value, state):
        validator = OneOf(PASSWORD_POLICY_NAMES, messages={'notIn': 'Unknown password policy'})
        return validator.to_python(value, state)


class Password(FormValidator):
    """
    Не используем min и max атрибуты validators.String, так как требуется выдавать все найденные ошибки сразу
    """
    messages = {
        'prohibitedSymbols': _('Password has prohibited symbols'),
        'tooShort': _('Password has less %(min)i symbols'),
        'tooLong': _('Password has more %(max)i symbols'),
        'weak': _('Password is not strong'),
        'likeLogin': _('Password is the same as login'),
        'likeOldPassword': _('Password is the same as old one'),
        'foundInHistory': _('Password found in history'),
        'likePhoneNumber': _('Password is the same as phone number'),
    }
    not_empty = True

    def __init__(
        self,
        password_field='password',
        login_field='login',
        policy_name_field='policy',
        old_password_hash_field='old_password_hash',
        uid_field='uid',
        country_field='country',
        phone_number_field='phone_number',
        emails_field='emails',
        required_check_password_history=False,
        required_check_password_pwned=True,
        required_password_quality=True,
        **kwargs
    ):
        self._password_field = password_field
        self._login_field = login_field
        self._policy_name_field = policy_name_field
        self._old_password_hash_field = old_password_hash_field
        self._uid_field = uid_field
        self._phone_number_field = phone_number_field
        self._country_field = country_field
        self._emails_field = emails_field
        self._required_check_password_history = required_check_password_history
        self._required_check_password_pwned = required_check_password_pwned
        self._required_password_quality = required_password_quality
        super(Password, self).__init__(**kwargs)

    def _to_python(self, value_dict, state):
        password = value_dict.get(self._password_field)
        if password is None:
            return value_dict

        new_value_dict = dict(value_dict)
        login = value_dict.get(self._login_field)
        login, email = (raw_login_from_email(login), login) if login and '@' in login else (login, None)
        emails = value_dict.get(self._emails_field)
        # Если емейлов нет (регистрация), построим дефолтный список
        emails = emails or set()
        if email:
            emails.add(email)
        old_password_hash = value_dict.get(self._old_password_hash_field)
        uid = value_dict.get(self._uid_field)
        policy_name = value_dict.get(self._policy_name_field, 'basic')
        policy = get_policy(policy_name)

        country = value_dict.get(self._country_field)
        phone_number = parse_phone_number(value_dict.get(self._phone_number_field), country)

        error_list_messages = []

        if len(password) < policy.min_length:
            error_list_messages.append('tooShort')

        if len(password) > policy.max_length:
            error_list_messages.append('tooLong')

        if policy.prohibited_symbols_re.search(password):
            error_list_messages.append('prohibitedSymbols')

        if login and normalize_login(login) == normalize_login(password):
            error_list_messages.append('likeLogin')
        else:
            password_as_email = normalize_email(password)
            for email in emails:
                normalized_email = normalize_email(email)
                if normalized_email == password_as_email:
                    error_list_messages.append('likeLogin')
                    break

        if phone_number:
            if phone_number.is_similar_to(password, country):
                error_list_messages.append('likePhoneNumber')

        if password and not error_list_messages:
            if old_password_hash and (
                (':' not in old_password_hash and check_password_equals_sha256_hash(password, old_password_hash)) or
                (':' in old_password_hash and check_password_equals_hash(password, old_password_hash, uid))
            ):
                error_list_messages.append('likeOldPassword')
            elif self._required_check_password_history:
                found_in_history = check_password_history(
                    password,
                    uid,
                    policy.search_depth,
                    reason=policy.reason,
                )
                if found_in_history:
                    error_list_messages.append('foundInHistory')

        if password and not error_list_messages and self._required_password_quality:
            words = []
            if login:
                words.append(login)
            words.extend(emails)

            quality = get_password_qualifier().get_quality(
                password,
                words=words,
                subwords=words,
            )
            quality_value = quality['quality']
            new_value_dict['quality'] = quality_value
            if state:
                state.password_quality = quality
            if quality_value < policy.min_quality:
                error_list_messages.append('weak')
            elif quality_value < policy.middle_quality:
                new_value_dict['lt_middle_quality'] = True

        login_in_shakur_whitelist = is_login_in_shakur_whitelist(login or '')
        if (
            password and
            not error_list_messages and
            not login_in_shakur_whitelist and
            settings.CHECK_SHAKUR_DURING_PASSWORD_VALIDATION and
            self._required_check_password_pwned
        ):
            is_password_pwned = check_password_is_pwned(password)
            if is_password_pwned:
                error_list_messages.append('weak')

        if error_list_messages:
            kw = {'min': policy.min_length, 'max': policy.max_length}
            error_list = [Invalid(self.message(msg_key, state, **kw), value_dict, state)
                          for msg_key in error_list_messages]
            raise Invalid(self.message(error_list_messages[0], state, **kw),
                          value_dict,
                          state,
                          error_dict={self._password_field: Invalid(self.message(error_list_messages[0], state, **kw),
                                                                    value_dict,
                                                                    state,
                                                                    error_list=error_list)})

        return new_value_dict


class PasswordHash(FancyValidator):
    strip = True
    all_hashes = {
        PASSWORD_ENCODING_VERSION_MD5_CRYPT: MD5_CRYPT_REGEX,
        PASSWORD_ENCODING_VERSION_MD5_HEX: MD5_RAW_REGEX,
        PASSWORD_ENCODING_VERSION_MD5_CRYPT_ARGON: MD5_CRYPT_REGEX,
        PASSWORD_ENCODING_VERSION_RAW_MD5_ARGON: MD5_RAW_REGEX,
    }
    allowed_hash_versions = [PASSWORD_ENCODING_VERSION_MD5_CRYPT]  # упорядоченный по убыванию приоритета список
    messages = {
        'invalid': 'Invalid password hash format',
    }

    def __init__(self, *args, **kwargs):
        super(PasswordHash, self).__init__(*args, **kwargs)
        for hash_version in self.allowed_hash_versions:
            if hash_version not in self.all_hashes:
                raise NotImplementedError('Unsupported hash version: %s' % hash_version)

    def _to_python(self, value, state):
        for hash_version in self.allowed_hash_versions:
            regex = self.all_hashes[hash_version]
            if regex.match(value):
                return hash_version, value

        raise Invalid(self.message('invalid', state), value, state)


class PasswordForAuth(String):
    """
    Валидатор пароля для авторизации.
    Должен падать в тех случаях, когда ЧЯ заведомо сочтет пароль неправильным
    """
    max = MAX_PASSWORD_LENGTH

    messages = {
        'notMatched': 'Password will not match',
    }

    def __init__(self, password_field='password', **kwargs):
        self._password_field = password_field
        super(PasswordForAuth, self).__init__(**kwargs)

    def validate_other(self, value, state):
        try:
            return super(PasswordForAuth, self).validate_other(value, state)
        except Invalid:
            raise Invalid(
                self.message('notMatched', state),
                value,
                state,
                error_dict={
                    self._password_field: Invalid(
                        self.message('notMatched', state),
                        value,
                        state,
                    ),
                },
            )
